CREATE TABLE lagerorte (
  lgo_name             VARCHAR(50) PRIMARY KEY,
  lgo_desc             TEXT,
  lgo_verfgbar         BOOLEAN DEFAULT true,           -- Lagerort geht in Verfügbarkeit ein.
  lgo_werthg           BOOLEAN DEFAULT true,           -- Lagerort ist werthaltig.
  lgo_beistell         BOOLEAN DEFAULT false,          -- Lagerort ist für Beistellmaterial.
  lgo_sperr            BOOLEAN NOT NULL DEFAULT false, -- Lagerort wird immer automatisch gesperrt.
  lgo_web              BOOLEAN NOT NULL DEFAULT false, -- Lagerzugang auf diesen Lagerort löst WEB aus.
  lgo_nodelete         BOOLEAN NOT NULL DEFAULT false, -- Lagerort bleibt erhalten und Eintrag in lag wird nicht gelöscht bei Menge 0.
  -- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Lager
  lgo_pflicht          BOOLEAN NOT NULL DEFAULT false, -- Wenn Standardlagerort im Artikel definiert und 'Zwang' vergeben ist, darf Artikel nur auf diesen Lagerort.
  lgo_pflichtfrei      BOOLEAN NOT NULL DEFAULT false, -- Zwang wird nicht berücksichtigt. Artikel mit 'Zwangslagerort' müssen auf Sonderlagerorte wie 'WE' (Wareneingangsprüfugn) oder 'QAB' (Qualitätsprüfung) gebucht werden können.
  lgo_ks               varchar(9) REFERENCES ksv (ks_abt) ON UPDATE CASCADE, -- Verlinkung zur Kostenstelle (Lagerort ist Standardlagerort dieser Kostenstelle)
  -- System (tables__generate_missing_fields)
  lgo_ad_krz_standort  varchar(30) NOT NULL DEFAULT '#' REFERENCES adressen_keys ON UPDATE CASCADE, -- Adress-Kurzname, abweichende Adresse des jeweiligen Lagerortes
  dbrid                VARCHAR(32) NOT NULL DEFAULT nextval('db_id_seq'),
  insert_date          DATE,
  insert_by            VARCHAR(32),
  modified_by          VARCHAR(32),
  modified_date        TIMESTAMP(0)
);

-- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
CREATE TRIGGER lagerorte_set_modified
  BEFORE INSERT OR UPDATE
  ON lagerorte
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();
--

--
CREATE OR REPLACE FUNCTION lagerorte__b_iud() RETURNS TRIGGER AS $$
  DECLARE old_lgo_name VARCHAR;
  BEGIN
    IF TG_OP = 'DELETE' THEN
        --Fall 1: zusammengesetzter Lagerort 'H1 | COMP ATA 12-10' soll entfernt werden
        IF (SELECT true FROM lagerortzuo WHERE (TLager.lag__lagort__bereich__split(old.lgo_name)).lagbereich = lgz_lgb_name
            AND (TLager.lag__lagort__bereich__split(old.lgo_name)).lagort = lgz_lgo_name)
        THEN
            RAISE EXCEPTION 'xtt25144'; --Achtung, es ist eine Zuordnung von Lagerbereich und Standardlagerort vorhanden. Entfernen Sie diese zuerst.
        END IF;
        --Fall 2: GrundStandard-Lagerort 'COMP ATA 12-10' soll entfernt werden
        IF (SELECT true FROM lagerortzuo WHERE (TLager.lag__lagort__bereich__split(old.lgo_name)).lagbereich IS NULL
            AND (TLager.lag__lagort__bereich__split(old.lgo_name)).lagort = lgz_lgo_name LIMIT 1)
        THEN
            RAISE EXCEPTION 'xtt25144'; --Könnte gesonderte Fehlermeldung erhalten
        END IF;

        RETURN old;
    ELSE -- INSERT, UPDATE
        new.lgo_name = UPPER(new.lgo_name);

        IF TG_OP = 'UPDATE' THEN
            old_lgo_name:= old.lgo_name;
        ELSE
            old_lgo_name:= new.lgo_name;
        END IF;

        -- Fehlerbehandlung
        IF new.lgo_name ~ '%' THEN -- Lagerort enthält %
            IF new.lgo_name = '%' THEN -- Nur % ist nicht erlaubt.
                RAISE EXCEPTION '%', lang_text(20103);
            END IF;

            IF new.lgo_name ~ '%.+' THEN -- Lediglich ein % am Ende ist erlaubt.
                RAISE EXCEPTION '%', lang_text(16609);
            END IF;

            -- widersprüchliche Konfigurationen mit % ausschließen (z.B. 'LO1%' und 'LO%' nicht erlaubt, da 'LO11' beide treffen würde)
            IF EXISTS(SELECT true FROM lagerorte WHERE lgo_name ~ '%' AND lgo_name <> old_lgo_name AND (lgo_name LIKE new.lgo_name OR new.lgo_name LIKE lgo_name)) THEN
                RAISE EXCEPTION '%', lang_text(16610) || E'\n\n' ||
                                     (SELECT string_agg(lgo_name, E'\n' ORDER BY lgo_name) FROM lagerorte WHERE lgo_name ~ '%' AND lgo_name <> old_lgo_name AND (lgo_name LIKE new.lgo_name OR new.lgo_name LIKE lgo_name));
            END IF;
        END IF;
        --

        RETURN new;
    END IF;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerorte__b_iud
    BEFORE INSERT OR UPDATE OR DELETE
    ON lagerorte
    FOR EACH ROW
    EXECUTE PROCEDURE lagerorte__b_iud();
--

--
CREATE OR REPLACE FUNCTION lagerorte__a_iu__lgo_sperr() RETURNS trigger AS $$
  BEGIN
      -- Automatisch alle Lagerorte sperren, wenn dies in der Standardlagerort-Konfiguration gesetzt wird.
      -- Entfernen der Konfig. gibt nicht automatisch alles frei, nur Sperren findet automatisch statt.


      UPDATE lag
         SET lg_sperr = true
       WHERE lg_ort LIKE new.lgo_name
         AND NOT lg_sperr
      ;


      RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerorte__a_iu__lgo_sperr
    AFTER INSERT OR UPDATE
    OF lgo_sperr
    ON lagerorte
    FOR EACH ROW
    WHEN ( new.lgo_sperr )
    EXECUTE PROCEDURE lagerorte__a_iu__lgo_sperr();
--

--
CREATE OR REPLACE FUNCTION lagerorte__a_iu__lgo_verfgbar() RETURNS trigger AS $$
  BEGIN
      -- Autom. Bestandsabgleich für Artikel an zug. Lagerorten
      -- bei Ändern der Verfügbarkeit an Standardlagerort-Konfiguration.


      PERFORM tartikel.bestand_abgleich_intern( lg_aknr )
         FROM lag
        WHERE lg_ort LIKE new.lgo_name
      ;


      RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerorte__a_iu__lgo_verfgbar
    AFTER INSERT OR UPDATE
    OF lgo_verfgbar
    ON lagerorte
    FOR EACH ROW
    EXECUTE PROCEDURE lagerorte__a_iu__lgo_verfgbar();
--

--
CREATE OR REPLACE FUNCTION lagerorte__a_u_pflicht() RETURNS TRIGGER AS $$
  DECLARE r RECORD;
  BEGIN
    IF (COALESCE((TLager.lag__lagort__bereich__split(new.lgo_name)).lagbereich, '') = '') THEN
        --Wir aktualisieren den Lagerort ohne Bereich > UPDATE alle zugehörigen Lagerorte mit Bereich
        PERFORM TSystem.Settings__Set('LagortOhneBereich', 'T'); --Setting um sicherzustellen, dass von hier nicht noch weiter in den ELSE Zweig gegangen wird
        FOR r IN SELECT lgz_lgb_name FROM lagerortzuo WHERE lgz_lgo_name=new.lgo_name LOOP
            UPDATE lagerorte SET lgo_pflicht=new.lgo_pflicht, lgo_pflichtfrei=new.lgo_pflichtfrei
            WHERE lgo_name=(SELECT TLager.lag__lagort__bereich__generate(r.lgz_lgb_name, new.lgo_name));
        END LOOP;
    ELSE --Wir aktualisieren den Lagerort,der den Bereich enthält > Meldung
        IF TSystem.Settings__Get('LagortOhneBereich')='F' THEN
            RAISE EXCEPTION 'xtt25142';
        END IF;
    END IF;
    PERFORM TSystem.Settings__Set('LagortOhneBereich', 'F');
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerorte__a_u_pflicht
    AFTER UPDATE
    OF lgo_pflicht, lgo_pflichtfrei
    ON lagerorte
    FOR EACH ROW
    EXECUTE PROCEDURE lagerorte__a_u_pflicht();
--

-- Funktion zur Ermittlung der Einstellungen eines Standardlagerortes anhand LIKE-Vergleich.
CREATE OR REPLACE FUNCTION lagerorte__get_setup(
    IN in_lagerort      VARCHAR,
    OUT _name           VARCHAR, -- Standardlagerort (ggf. mit Wildcard)
    OUT _verfgbar       BOOLEAN, -- Verfügbarkeit
    OUT _werthg         BOOLEAN, -- Werthaltigkeit
    OUT _beistell       BOOLEAN, -- Beistell-LO
    OUT _sperr          BOOLEAN, -- Sperrlager
    OUT _web            BOOLEAN, -- LO verlangt WEB
    OUT _nodelete       BOOLEAN, -- LO erhalten
    OUT _pflicht        BOOLEAN, -- StandardLO-Zwang
    OUT _pflichtfrei    BOOLEAN  -- StandardLO-Zwang wird nicht berücksichtigt
  ) RETURNS RECORD AS $$
  BEGIN
    SELECT lgo_name, lgo_verfgbar, lgo_werthg, lgo_beistell, lgo_sperr, lgo_web, lgo_nodelete, lgo_pflicht, lgo_pflichtfrei
    INTO   _name,    _verfgbar,    _werthg,    _beistell,    _sperr,    _web,    _nodelete,    _pflicht,    _pflichtfrei
    FROM lagerorte WHERE in_lagerort LIKE lgo_name -- LIKE-Suche über Standardlagerorte
    -- Bei mehreren Treffern aufgrund LIKE und Konfiguration beim Kunden nach defnierter Reihenfolge den passenden Standardlagprt zurückgeben. Achtung! ORDER BY BOOLEAN
    ORDER BY lgo_name <> in_lagerort, -- gleichen String bevorzugen ('LO1' ist definiert, vor 'LO%' nehmen)
        NOT lgo_verfgbar, lgo_sperr, NOT lgo_werthg, -- bevorzugen: verfügbar, nicht gesperrt, werthaltig
        lgo_name -- im Notfall nach PKey, damit eindeutig
    LIMIT 1;

    -- Fallback per COALESCE(..., true/false) nicht machen, da COALESCE im Einzelfall entschieden werden muss, z.B. siehe TArtikel.art__lag__lg_anztot__get__nverfueg.

    RETURN;
  END $$ LANGUAGE plpgsql STABLE STRICT;
--

CREATE OR REPLACE FUNCTION tlager.wendat_get_aufteilung(
    IN ldid integer,
    IN menge numeric,
    IN los_menge numeric,
    OUT w_pos integer,
    OUT w_stk numeric)
  RETURNS SETOF record AS $$
DECLARE
  source RECORD;
  pos INTEGER;
  stk NUMERIC;
BEGIN
  SELECT * INTO source FROM ldsdok WHERE ld_id = ldid LIMIT 1;
  --wenn eingegebene Menge nicht positiv ist, Menge-offen nehmen
  IF COALESCE(menge, 0) <= 0 THEN
    menge = source.ld_stk - tartikel.me__menge_uf1__in__menge(source.ld_mce, source.ld_stkl);
    --wenn Menge-offen auch nicht positiv ist, exit
    IF COALESCE(menge, 0) <= 0 THEN
      RETURN;
    END IF;
  END IF;
  --wenn Losmenge nicht positiv ist, Einkaufslosmenge nehmen
  IF COALESCE(los_menge, 0) <= 0 THEN
    los_menge = source.ld_eklos;
    --wenn Einkaufslosmenge auch nicht positiv ist, exit
    IF COALESCE(los_menge, 0) <= 0 THEN
      RETURN;
    END IF;
  END IF;

  --wenn mehr als 100 Datensätze raus kommen => nur 1 zurückgeben, damit Nutzer los_menge anpassen kann
  IF ((menge / los_menge) > 100) THEN
    los_menge = menge;
  END IF;

  pos = 1;
  WHILE menge > 0 LOOP
    w_pos = pos;
    IF menge >= los_menge THEN
      w_stk = los_menge;
      menge = menge - los_menge;
    ELSE
      w_stk = menge;
      menge = 0;
    END IF;
    RETURN NEXT;
    pos = pos + 1;
  END LOOP;
  RETURN;
END $$ LANGUAGE plpgsql VOLATILE;
--
CREATE OR REPLACE FUNCTION lgort_dosperr(lgort VARCHAR) RETURNS BOOL AS $$
  BEGIN
    RETURN COALESCE((lagerorte__get_setup(lgort))._sperr, false);
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION lgort_doweb(lgort VARCHAR) RETURNS BOOL AS $$
  BEGIN
    RETURN COALESCE((lagerorte__get_setup(lgort))._web, false);
  END $$ LANGUAGE plpgsql STABLE;
--

-- Lagerkonfiguration für Artikel (Artikelstamm, Tab Lager)
CREATE TABLE lagartikelkonf (
  lgoa_aknr            VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
  lgoa_lgo_name        VARCHAR(50) REFERENCES lagerorte ON UPDATE CASCADE,
  lgoa_min             NUMERIC(12,4) NOT NULL DEFAULT 0,
  lgoa_melde           NUMERIC(12,4),
  lgoa_soll            NUMERIC(12,4)
);

--
CREATE OR REPLACE FUNCTION lagartikelkonf__a_iud() RETURNS TRIGGER AS $$
  DECLARE min NUMERIC(12,4);
          melde  NUMERIC(12,4);
          soll NUMERIC(12,4);
          aknr VARCHAR(50);
  BEGIN
    IF tg_op='INSERT' THEN
        aknr:=new.lgoa_aknr;
    ELSE
        aknr:=old.lgoa_aknr;
    END IF;
    --
    SELECT SUM(lgoa_min), SUM(lgoa_melde), SUM(lgoa_soll) INTO min, melde, soll FROM lagartikelkonf WHERE lgoa_aknr=aknr;
    UPDATE art SET ak_min=COALESCE(min,0), ak_melde=COALESCE(melde,0), ak_soll=COALESCE(soll,0) WHERE aknr=ak_nr;
    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagartikelkonf__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON lagartikelkonf
    FOR EACH ROW
    EXECUTE PROCEDURE lagartikelkonf__a_iud();
--

--Nach Zuweisung von Lagerort-Pflicht, Synchronhalten in beiden Einträgen  #6305
--Es soll immer der Standardlagerort ohne Bereichszuordnung gesetzt werden > der Status wird in die Lagerorte mit Bereichszuordnung vererbt
-- Lagerbereiche Grunddaten #6259
CREATE TABLE lagerbereich (
  lgb_name      VARCHAR(50) PRIMARY KEY,
  lgb_descr     VARCHAR(100),
  lgb_zutxt     TEXT
);


-- Predefines: lahgerbereich UE für unvollendete Artikel welche dennoch mit Arbeitsgang im Lager liegen

-- Verknüpfung von Lagerbereich und Standardlagerort #6259
CREATE TABLE lagerortzuo (
  lgz_id            SERIAL PRIMARY KEY,
  lgz_lgo_name      VARCHAR(50) NOT NULL REFERENCES lagerorte ON UPDATE CASCADE ON DELETE CASCADE,
  lgz_lgb_name      VARCHAR(50) NOT NULL REFERENCES lagerbereich ON UPDATE CASCADE ON DELETE CASCADE
);

--Wenn neuer Lagerbereich zum Standardlagerort definiert wird, erzeuge einen neuen Standardlagerort mit dem Bereich im Ident 'H2 | ATA 47-11'
CREATE OR REPLACE FUNCTION lagerortzuo__a_i() RETURNS TRIGGER AS $$
  DECLARE _rec lagerorte;
  BEGIN
    SELECT * INTO _rec FROM lagerorte WHERE lgo_name = new.lgz_lgo_name;
    _rec.lgo_name := TLager.lag__lagort__bereich__generate(new.lgz_lgb_name, new.lgz_lgo_name);
    _rec.dbrid    := nextval('db_id_seq');

    -- Lager UE: default Lagerbereich, der Vorgaben an die zugewiesenen Lagerorte weitergibt
    IF new.lgz_lgb_name = 'UE' THEN
       _rec.lgo_werthg   := false;
       _rec.lgo_verfgbar := false;
       _rec.lgo_sperr    := true;
       _rec.lgo_nodelete := false;
    END IF;

    INSERT INTO lagerorte VALUES ((_rec).*);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerortzuo__a_i
    AFTER INSERT
    ON lagerortzuo
    FOR EACH ROW
    EXECUTE PROCEDURE lagerortzuo__a_i();
--

--
CREATE OR REPLACE FUNCTION lagerortzuo__a_d() RETURNS TRIGGER AS $$
  BEGIN
    DELETE FROM lagerorte WHERE lgo_name=TLager.lag__lagort__bereich__generate(old.lgz_lgb_name, old.lgz_lgo_name);
    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerortzuo__a_d
    AFTER DELETE
    ON lagerortzuo
    FOR EACH ROW
    EXECUTE PROCEDURE lagerortzuo__a_d();
--

--
CREATE TABLE lag (
  lg_id                SERIAL NOT NULL PRIMARY KEY,
  lg_aknr              VARCHAR(40) NOT NULL CONSTRAINT xtt5000 NOT NULL REFERENCES art ON UPDATE CASCADE,
  lg_ort               VARCHAR(50) NOT NULL CONSTRAINT xtt16611__lg_ort CHECK (strpos(lg_ort, '%') = 0),
  lg_anztot            NUMERIC(12,4) NOT NULL DEFAULT 0,               --bestand am lagerort
  lg_anzist            NUMERIC(12,4) NOT NULL DEFAULT 0,               --inventurbestand
  lg_mce               INTEGER REFERENCES artmgc ON DELETE SET NULL,
  lg_lagzudat          DATE DEFAULT current_date,                      --alter am lagerort/zugangsdatzn für FIFO
  lg_konsDat           DATE,                                           --Konservierungsdatum (LOLL : Nutzer tragen manuell ein, wann Bauteile konserviert (Korrosionsschutz) wurden)
  lg_konsEnde          DATE,                                           -- Ablauf der Konservierung
  lg_zaedat            DATE,
  lg_chnr              VARCHAR(50) NOT NULL,
  lg_dim1              NUMERIC(12,4) NOT NULL DEFAULT 0,
  lg_dim2              NUMERIC(12,4) NOT NULL DEFAULT 0,
  lg_dim3              NUMERIC(12,4) NOT NULL DEFAULT 0,
  lg_w_wen             INTEGER, --=> X TableConstraints: REFERENCES wendat ON UPDATE CASCADE ON DELETE SET NULL
  lg_ld_id             INTEGER REFERENCES ldsdok ON UPDATE CASCADE ON DELETE SET NULL,
  lg_sperr             BOOLEAN DEFAULT FALSE,
  lg_regalsta          INTEGER,
  --lg_internalchange  BOOLEAN DEFAULT FALSE
  lg_txt               TEXT         -- Lagerbuchung Bemerkung #9905
);

--Indizes
    CREATE UNIQUE INDEX uniquelag ON lag (lg_aknr, lg_ort, lg_chnr, lg_dim1, lg_dim2, lg_dim3);
    CREATE INDEX lag_lg_ort ON lag(lg_ort);
    CREATE INDEX lag_aknr_like ON lag(lg_aknr varchar_pattern_ops);
    CREATE INDEX lag_lgort_like ON lag(lg_ort varchar_pattern_ops);
    CREATE INDEX lag_lgchnr_like ON lag(lg_chnr varchar_pattern_ops);
--

--

CREATE OR REPLACE FUNCTION lag__b_iu() RETURNS TRIGGER AS $$
  DECLARE
      _iv_start                             timestamp;
      _il_anz_ist                           numeric;
      _wendat_anz                           numeric;
      _lifsch_anz                           numeric;
      _inv__konkurrierende_inventur__exists boolean;
      _inv__konkurrierende_inventur__list   varchar[];
  BEGIN

      new.lg_ort  := upper( trim( new.lg_ort ) );
      new.lg_chnr := upper( trim( new.lg_chnr ) );

      -- Vorab prüfen, ob eine laufende Inventur ohne Eintrag vom Positions-Zähldatum vorhanden ist. Dann sind keine Buchungen möglich.
      SELECT  inv__konkurrierende_inventur__exists,  inv__konkurrierende_inventur__list
        INTO  _inv__konkurrierende_inventur__exists, _inv__konkurrierende_inventur__list
        FROM TLager.lag__inv__konkurrierende_inventur__exists( new, true )  -- lag version
      ;

      -- Keine Lagerbewegung möglich, da aktuell eine Inventur läuft und die Zählwerte noch nicht eingetragen wurden. Laufende Inventur(en):
      IF _inv__konkurrierende_inventur__exists THEN
          RAISE EXCEPTION '%', format( lang_text(32608), array_to_string( _inv__konkurrierende_inventur__list, ', ' ) );
      END IF;

      -- Lager-UE: bei Zubuchung/umbuchen sperren. Validieren das Daten "valide" bleiben: ehemaliger UE lager Satz darf nicht zum normalen werden. (lg_w_wen entfernt)
        IF new.lg_w_wen IS NOT NULL THEN
           IF w_a2_id IS NOT NULL FROM wendat WHERE w_wen = new.lg_w_wen THEN
              IF (lagerorte__get_setup(new.lg_ort))._verfgbar THEN
                 RAISE EXCEPTION 'UE-Lager darf nicht verfügbar sein und ist immer gesperrt. Der Lagerort ist aber als verfügbar definiert (Standard-Lagerort)';
              END IF;
              new.lg_sperr := true;
           END IF;
        END IF;
        --
        IF tg_op = 'UPDATE' THEN
           IF old.lg_w_wen IS NOT NULL AND new.lg_w_wen IS NULL THEN
              IF w_a2_id IS NOT NULL FROM wendat WHERE w_wen = old.lg_w_wen THEN
                 RAISE EXCEPTION 'UE-Lagerbezug wareneingang darf nicht durch Vermischung aufgehoben werden: "old.lg_w_wen IS NOT NULL AND new.lg_w_wen IS NULL"';
              END IF;
           END IF;
        END IF;
      -- Validierung UE Lager

      --Konservierundsatum sinnig prüfen
      IF new.lg_konsDat > new.lg_konsEnde THEN
         new.lg_konsEnde := null;
      END IF;

      IF TG_OP='UPDATE' THEN

          --Bestellbezug entfernen, wenn Lagerort leer, damit neuer Lagerzugang auch wieder einen Bestellbezug vergeben kann!
          IF new.lg_anztot <= 0 THEN
             new.lg_w_wen := null;
             new.lg_ld_id := null;
          END IF;

          -- Datum neu setzen bei Lagorten mit 0-Bestand, wenn neue, positive Menge (Standardlagerorte)
          IF new.lg_anztot > 0 AND old.lg_anztot <= 0 THEN
             new.lg_lagzudat := current_date;
          END IF;
      END IF;

      -- neuer Lagerort darf nicht '' sein. Nur bei INSERT, sonst Lagerkorrekturen auf Lagerorte '' nach Einführung der Einstellung sonst nicht möglich.
      IF
             TG_OP = 'INSERT'
         AND new.lg_ort = ''
         AND TSystem.Settings__GetBool( 'NoEmptyLagO' )
      THEN
          RAISE EXCEPTION 'xtt16245';
      END IF;


      IF -- neuer Lagerort: Benutzerrechte prüfen
            -- kein Hochregal, oder QS-Lager
            NOT ( new.lg_ort LIKE '%-%-%' )
            -- Wenn Lagerort anlegen mit Nutzerrechten geprüft wird
        AND EXISTS( SELECT * FROM pg_group WHERE groname LIKE 'SYS.Lager-Lagerort' )
      THEN

          IF
                 -- Prüfen ob Nutzer notwendige Rechte zur Anlage hat
                 NOT current_user IN (
                    SELECT usename
                    FROM pg_group, pg_user
                    WHERE usesysid = any( grolist )
                      AND groname = 'SYS.Lager-Lagerort'
                 )
                 -- Lagerort ist bereits vorhanden
             AND NOT EXISTS( SELECT lg_ort FROM lag WHERE lg_ort = new.lg_ort )
                 -- Lagerort ist vordefiniert in Tabelle "lagerorte"
             AND NOT EXISTS( SELECT lgo_name FROM lagerorte WHERE lgo_name = new.lg_ort )
                 -- Werkzeugumbuchungen auf Kostenstellen
             AND NOT EXISTS( SELECT ks_abt FROM ksv WHERE ks_abt = new.lg_ort )
          THEN
              RAISE EXCEPTION 'xtt10204 - "%"', new.lg_ort;
          END IF;
      END IF;

      IF
            -- weil sonst jede änderung am lagerplatz das sperrkennzeichen setzen würde, freigeben ist damit fast nicht mehr möglich
            TG_OP = 'INSERT'
        AND (
                 lgort_dosperr( new.lg_ort )
              OR ( SELECT ak_sperrlag FROM art WHERE ak_nr = new.lg_aknr )
            )
      THEN
          new.lg_sperr := true;
      END IF;

      IF new.lg_mce IS NULL THEN
          new.lg_mce := tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( new.lg_aknr );
      END IF;

      IF new.lg_chnr = '' THEN

          -- Chargennummer erforderlich
          IF ak_chnrreq FROM art WHERE ak_nr = new.lg_aknr THEN
              RAISE EXCEPTION 'xtt11066 "!!Charge"';
          END IF;

      -- Konflikt Standardlagerort und Chargennummernpflicht, siehe Trigger art__a_iu__akslort
      ELSEIF new.lg_chnr = 'ART__A_IU__AKSLORT' THEN
          new.lg_chnr := '';
      END IF;

      RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lag__b_iu
    BEFORE INSERT OR UPDATE
    ON lag
    FOR EACH ROW
    EXECUTE PROCEDURE lag__b_iu();
--

-- Zwangslagerorte #6305 - Als eigener Trigger wegen Umbuchung. Prüfung des neuen Lagerortes auf Lagerortpflicht laut Artikel-Lagerort-Konfiguration.
CREATE OR REPLACE FUNCTION lag__b_iu__lgort() RETURNS TRIGGER AS $$
  DECLARE new_lagort VARCHAR;
          pflicht_lagerort VARCHAR;
  BEGIN
    new_lagort:= (TLager.lag__lagort__bereich__split(new.lg_ort)).lagort;
    IF NOT COALESCE((lagerorte__get_setup(new_lagort))._pflichtfrei, false) THEN -- neuer Lagerort ist nicht "pflichtfrei" (Zwang wird nicht berücksichtigt)
        -- Zwangslagerort laut Artikelstamm ermitteln (lgo_pflicht) bzw. NULL.
        pflicht_lagerort:= (SELECT ak_slort FROM art JOIN LATERAL lagerorte__get_setup(ak_slort) ON ak_slort IS NOT NULL WHERE ak_nr = new.lg_aknr AND _pflicht);

        IF new_lagort <> COALESCE(pflicht_lagerort, new_lagort) THEN -- weicht der neue Lagerort vom Zwangslagerort ab, dann
            RAISE EXCEPTION '%', lang_text(25140); -- Lagerzugang aufgrund Lagerortzwang nicht möglich.
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lag__b_iu__lgort
    BEFORE INSERT OR UPDATE
    OF lg_ort
    ON lag
    FOR EACH ROW
    EXECUTE PROCEDURE lag__b_iu__lgort();
--

-- Verfügbarkeit und LAGERLOG und WEB erstellen
CREATE OR REPLACE FUNCTION lag__a_i() RETURNS TRIGGER AS $$
  DECLARE wwenRec RECORD;
          webnr   INTEGER;
  BEGIN
    IF current_user='syncro' THEN
        RETURN new;
    END IF;
    --Bedarfsberechnung und Bestandsabgleich in __a_iu
    -- Bei Einlagern auf einen Lagerort, der WEB anlegen soll, legen wir den jetzt an, falls es nicht schon einen vom Wareneingang gibt.
    -- Passiert z.Bsp. bei Wareneingang auf Lagerort "Zeugniss fehlt" und umbuchen auf ZLQ.
    IF lgort_doweb(new.lg_ort) THEN
        SELECT w_wen, w_l_krz INTO wwenRec FROM wendat WHERE
          w_aknr = new.lg_aknr
          AND w_lgchnr = new.lg_chnr
          AND NOT EXISTS(SELECT true FROM wareneingangskontrolle WHERE wek_w_wen = w_wen)
        ORDER BY w_wen DESC LIMIT 1;

        IF (wwenRec.w_wen IS NOT NULL) THEN
            SELECT getnumcirclenr('web')::INTEGER INTO webnr;
            INSERT INTO wareneingangskontrolle (wek_nr,wek_w_wen,wek_ak_nr,wek_l_krz,wek_dat,wek_q_nr)
                VALUES (webnr, wwenrec.w_wen, new.lg_aknr, wwenrec.w_l_krz, Current_Date,
                (SELECT q_nr FROM qab WHERE q_w_wen = wwenrec.w_wen AND NOT q_def ORDER BY q_nr DESC LIMIT 1));
            -- Durch das automatische QAB erstellen kann es prinipiell mehrere geben, aber immer nur einen offenen QAB.
            PERFORM CopyArtTests('wareneingangskontrolle',webnr,NULL); --Artikelpruefungen fuer den Artikel noch umkopieren

            PERFORM PRODAT_HINT(lang_text(16533) || E'\n\n' || webnr);
        END IF;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lag__a_i
    AFTER INSERT
    ON lag
    FOR EACH ROW
    EXECUTE PROCEDURE lag__a_i();
--

--
CREATE OR REPLACE FUNCTION lag__a_u() RETURNS TRIGGER AS $$
  DECLARE PM VARCHAR(1);
  BEGIN
    -- Bedarfsberechnung und Bestandsabgleich findet nicht hier sondern in __a_iu statt.
    -- Interne Lagerbewegungen (durch wendat, lifsch usw.) werden explizit dort selbst in Lagerlog geschrieben.

    IF ( TSystem.Settings__Get('lagerlog_'||current_user) = 'F' OR new.lg_regalsta IS DISTINCT FROM old.lg_regalsta ) THEN -- kein Logging
    ELSIF current_user <> 'syncro' THEN
        PM := '';
        IF new.lg_anztot > old.lg_anztot THEN   PM := '+';  END IF;
        IF new.lg_anztot < old.lg_anztot THEN   PM := '-';  END IF;

        --  Achtung, Sonderbehandlung für UE Lager in lagerlog__b_i
        PERFORM TLager.lagerlog__create( '==' || PM, new, old);
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lag__a_u
    AFTER UPDATE
    OF lg_aknr, lg_anztot, lg_ort, lg_chnr, lg_dim1, lg_dim2, lg_dim3, lg_sperr
    ON lag
    FOR EACH ROW
    EXECUTE PROCEDURE lag__a_u();
--

--
CREATE OR REPLACE FUNCTION lag__a_iu() RETURNS TRIGGER AS $$
  DECLARE slort varchar;
          isstandardlagerort bool;
          artisgueltig bool;
  BEGIN

    --Behandlung von Negativbeständen
    IF      (new.lg_ort <> 'MINUS')
        AND (new.lg_anztot < 0)
        AND TSystem.Settings__Get('no_neg_lag') = -1
    THEN -- Generell keine negativen Lagerbestände
        UPDATE lag SET lg_anztot = 0 WHERE dbrid = new.dbrid;--Lagerbestand auf 0 wenn keine negavtien zugelassen
        -- RETURN damit nicht rekursiv?
    END IF;

    -- Standardlagerort holen wenn Bestand <= 0, damit kein Standardlagerort automatisch gelöscht wird
    IF (new.lg_anztot <= 0) THEN
        --standardlagerort holen wenn artikel gültig
        SELECT ak_slort, coalesce(ak_auslauf, current_date) >= current_date
          INTO slort, artisgueltig
          FROM art
         WHERE ak_nr = new.lg_aknr
           AND coalesce(ak_auslauf, current_date) >= current_date
        ;

        IF slort IS NULL THEN
           isstandardlagerort := false; -- kann somit gelöscht werden
        ELSE
           isstandardlagerort := new.lg_ort LIKE slort;--man kann damit ein "%" im artikelstamm eintragen und ähnliche lager damit abfangen
        END IF;

        --
        IF         artisgueltig
           AND NOT isstandardlagerort
           AND     coalesce((lagerorte__get_setup(new.lg_ort))._nodelete, false)
        THEN -- wenn dieser Lagerort per Standard nicht gelöscht werden darf, dann auch nicht löschen und wie einen Standardlagerort behandeln
           isstandardlagerort := true;
        END IF;
    END IF;

    -- Automatischen Lagerort/Chargen löschen! (interne Lagerorte für automatisches Buchen)
    IF      (new.lg_anztot <= 0)
        AND ((new.lg_chnr LIKE 'ABK=%') OR (new.lg_ort LIKE 'ALAUTO'))
        AND (TSystem.Settings__GetBool('AutoDelAbkLO'))
        AND NOT isstandardlagerort --standardlagerorte nicht löschen
    THEN --Bestand 0 und automatisch Lagerorte löschen (Charge) wenn nicht Standardlagerort
        DELETE FROM lag
              WHERE dbrid = new.dbrid
                    --Lagerorte nicht löschen, da SN vorhandenden, siehe: https://redmine.prodat-sql.de/issues/16515#note-30
                    --würde der lag-Eintrag gelöscht, würde durch lagsernr__a_u__check_orphan auch der lagsernr-Einträg gelöscht werden, da ON DELETE SET NULL;
                    AND NOT EXISTS(SELECT true FROM lagsernr WHERE lgs_lg_id=new.lg_id)
        ;
    END IF;

    -- Negative / 0 Lagerorte löschen
    IF      new.lg_anztot <= 0
       AND (
               -- wenn mit Charge, lagerorte immer löschen, da charge nie wieder kommen sollte
               (coalesce(new.lg_chnr, '') <> '')
            OR -- wenn nicht lagerhaltig, immer automatisch löschen!
               (SELECT NOT ak_lag FROM art WHERE ak_nr = new.lg_aknr)
               -- trotz "nicht lagerartikel" sind lagerbuchungen möglich, auch die mengen werden korrekt berechnet.
            OR (    (TSystem.Settings__GetBool('DelEmptyLagO')) -- nochmal prüfen
                AND (new.lg_ort <> 'MINUS')
                AND NOT isstandardlagerort --standardlagerort nicht löschen
               )
           )
    THEN -- Bestand 0 und Lagerorte löschen (Ausnahme: Lagerort MINUS)
        IF NOT TSystem.Settings__GetBool('umbuch') THEN -- beim umbuchen darf nicht gelöscht werden!
           DELETE FROM lag  --automatischen Lagerort löschen!
                  WHERE dbrid = new.dbrid
                        --Lagerorte nicht löschen, da SN vorhanden, siehe: https://redmine.prodat-sql.de/issues/16515#note-30
                        --würde der lag-Eintrag gelöscht, würde durch lagsernr__a_u__check_orphan auch der lagsernr-Einträg gelöscht werden, da ON DELETE SET NULL;
                        AND NOT EXISTS(SELECT true FROM lagsernr WHERE lgs_lg_id=new.lg_id AND lgs_l_nr IS NULL)
           ;
        END IF;
    END IF;

    -- Automatischen Lagerort "MINUS" löschen, wenn ausgeglichen
    IF new.lg_anztot = 0 AND (new.lg_ort = 'MINUS') THEN -- bestand 0 und lagerorte löschen (Ausnahme: Lagerort MINUS)
       DELETE FROM lag WHERE dbrid = new.dbrid;
    END IF;

    -- Bedarfsberechnung
    IF TG_OP = 'UPDATE' THEN
       PERFORM tartikel.prepare_artikel_bedarf( old.lg_aknr, false );
    END IF;
    PERFORM tartikel.prepare_artikel_bedarf( new.lg_aknr, false );


    /*interne Lagerbewegungen (durch Wendat, Lifsch) werden durch diese in Lagerlog...*/
    IF tg_op='INSERT' AND current_user<>'syncro' THEN --ACHTUNG, steht hier und nicht im Before_Insert, da hier der Bestandsabgleich ausgelöst wird und dies zwingend vor dem Lagerlog Eintrag passieren muß!
        IF TSystem.Settings__Get('lagerlog_'||current_user)='F' THEN--kein Logging (wird durch Wendat-Trigger)
        ELSE
            PERFORM TLager.lagerlog__create( '++', new); --- #14773 lag__a_iu()
        END IF;
    END IF;

    -- Konservierungsdatum erforderlich
    IF (SELECT ak_konsreq FROM art WHERE ak_nr = new.lg_aknr) AND (new.lg_konsDat IS NULL /*OR new.lg_konsEnde IS NULL*/) THEN
      -- Konservierungsdatum erforderlich [Artikel= | Lagerort= | Charge= ]
      RAISE EXCEPTION '%', FORMAT('%s ' || E'\r\N' || '[%s=%s | %s=%s | %s=%s]', lang_text(17990), Lang_text(234), new.lg_aknr, lang_text(509), new.lg_ort, lang_text(28003),new.lg_chnr);
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lag__a_iu
    AFTER INSERT OR UPDATE
    ON lag
    FOR EACH ROW
    EXECUTE PROCEDURE lag__a_iu();
--

--
CREATE OR REPLACE FUNCTION lag__a_d() RETURNS TRIGGER AS $$
  DECLARE tot NUMERIC;
  BEGIN
    IF current_user='syncro' THEN
        RETURN old;
    END IF;
    --
    PERFORM tartikel.prepare_artikel_bedarf(old.lg_aknr, true);
    --
    IF old.lg_anztot=0 THEN --kein Eintrag in die Lagerlog, wenn nichts da war.
        RETURN old;
    END IF;
    --
    IF TSystem.Settings__Get('lagerlog_'||current_user)='F' THEN  --kein Logging, wenn absichtlich deaktiviert
    ELSE
        PERFORM TLager.lagerlog__create( '--', NULL, old); --- #14773 lag__a_d()
    END IF;

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lag__a_d
    AFTER DELETE
    ON lag
    FOR EACH ROW
    EXECUTE PROCEDURE lag__a_d();
--

--
CREATE OR REPLACE FUNCTION lag__as__a_iud() RETURNS TRIGGER AS $$
  BEGIN
    IF current_user='root' THEN
        RAISE NOTICE 'ENTER FUNCTION lag__as__a_iud() - %', currenttime();
    END IF;
    --
    PERFORM do_artikel_bedarf();
    --
    IF current_user='root' THEN
        RAISE NOTICE 'FULL EXIT FUNCTION lag__as__a_iud() - %', currenttime();
    END IF;
    --
    RETURN NULL;
  END $$ LANGUAGE plpgsql;

  CREATE CONSTRAINT TRIGGER lag__as__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    --OF lg_aknr, lg_anztot
    ON lag
    DEFERRABLE INITIALLY DEFERRED
    FOR EACH ROW
    EXECUTE PROCEDURE lag__as__a_iud();
--

--
CREATE SEQUENCE lag_umbuchid START WITH 10000;
--

-- Lagerort / Chargennummer wechseln
CREATE OR REPLACE FUNCTION tlager.lag__ort_chnr__lagerortwechsel(
    _aknr                 varchar, -- Artikel
    _oort                 varchar, -- alter Lagerort
    _ochnr                varchar, -- alte ChargenNr.
    _nort                 varchar, -- neuer Lagerort
    _nchnr                varchar, -- neue ChargenNr.
    _menge                numeric, -- Mengen
    _dim1                 integer   DEFAULT 0, -- Dimension
    _dim2                 integer   DEFAULT 0,
    _dim3                 integer   DEFAULT 0,
    -- Optionen
    _dellag               boolean   DEFAULT true, -- alten Lagerort bei Menge 0 löschen
    _autosernrwechsel     boolean   DEFAULT true, -- Seriennummern autom. mit Umbuchen.
    _umbuchid             integer   DEFAULT nextval('lag_umbuchid'),  -- Umbuchungs-ID
    _bemerk               text      DEFAULT NULL  -- Bemerkungstext aus Oberfläche
    )
    RETURNS integer
    AS $$
    DECLARE _nlgid        integer; -- ID neuer lagerort
            _olgid        integer; -- ID alter lagerort
            _oldlgwen     integer;
            _updlgwen     integer;
            _oldldid      integer;
            _updldid      integer;
            _oldKonsDat   date;
            _oldKonsEnde  date;
            _r            record;
            _lagzudat     date;
            _rows         integer;
            _hnnr         varchar;
            _umbuchsernr  varchar;
            _oldlgtxt     text;
            _check_bezug  boolean;
    BEGIN

        -- es gibt nichts umzubuchen
        IF coalesce(_menge, 0) = 0 THEN
           RETURN null;
        END IF;

        PERFORM TSystem.Settings__Set('umbuch', 't'); -- Umbuchungs-Flag zur Trigger-Steuerung

        -- Eingabefehler abfangen
        _aknr := upper(trim(_aknr));
        _oort := upper(trim(coalesce(_oort, '')));
        _ochnr:= upper(trim(coalesce(_ochnr, '')));
        _nort := upper(trim(coalesce(_nort, '')));
        _nchnr:= upper(trim(coalesce(_nchnr, '')));

        -- altes Datum und ID merken
        SELECT lg_id, lg_lagzudat, lg_w_wen, lg_ld_id, lg_konsDat, lg_konsEnde, lg_txt
          INTO _olgid, _lagzudat,  _oldlgwen, _oldldid, _oldKonsDat, _oldKonsEnde, _oldlgtxt
          FROM lag
         WHERE lg_aknr = _aknr
           AND lg_ort  = _oort
           AND lg_chnr = _ochnr
           AND lg_dim1 = _dim1
           AND lg_dim2 = _dim2
           AND lg_dim3 = _dim3;

        LOCK TABLE lagerlog IN EXCLUSIVE MODE;

        -- Bemerkung aus Umbuchung ergänzen
        IF _bemerk IS NOT NULL THEN
           _oldlgtxt := coalesce(_oldlgtxt || E'\n\n', '') || _bemerk;
        END IF;

        -- alten Lagerbestand runter
        UPDATE lag SET lg_anztot = lg_anztot - _menge WHERE lg_id = _olgid; -- Der Lagerort bleibt erhalten, da Umbuchungs-Flag gesetzt ist.
        -- Umbuchungs-ID setzen
        UPDATE lagerlog SET lo_umbuchid = _umbuchid WHERE lo_id = (SELECT max(lo_id) FROM lagerlog); -- letzter Eintrag ist aus Lag in Lagerlog

        -- neuen Lagerbestand rauf
        UPDATE lag
           SET lg_lagzudat  = least(lg_lagzudat, _lagzudat),
               lg_anztot    = lg_anztot + _menge,
               lg_w_wen     = CASE WHEN lg_anztot <= 0 THEN _oldlgwen    ELSE lg_w_wen END, -- Bezug erneut herstellen bei Bestand <=0 (Standardlagerort)
               lg_ld_id     = CASE WHEN lg_anztot <= 0 THEN _oldldid     ELSE lg_ld_id END, -- ebenso
               lg_konsDat   = CASE WHEN lg_anztot <= 0 THEN _oldKonsDat  ELSE least(lg_konsDat,  _oldKonsDat)  END,
               lg_konsEnde  = CASE WHEN lg_anztot <= 0 THEN _oldKonsEnde ELSE least(lg_konsEnde, _oldKonsEnde) END,
               lg_txt       = CASE WHEN lg_txt IS NULL THEN _oldlgtxt    ELSE lg_txt || coalesce(E'\n\n' || _oldlgtxt, '') END
         WHERE lg_aknr = _aknr
           AND lg_ort  = _nort
           AND lg_chnr = _nchnr
           AND lg_dim1 = _dim1
           AND lg_dim2 = _dim2
           AND lg_dim3 = _dim3
        RETURNING lg_id, lg_w_wen, lg_ld_id
        INTO      _nlgid, _updlgwen, _updldid;

        IF NOT found THEN -- den neuen Lagerort gibts noch nicht
            INSERT INTO lag (lg_aknr, lg_ort, lg_chnr, lg_anztot,
                             lg_dim1, lg_dim2, lg_dim3,
                             lg_lagzudat, lg_w_wen, lg_ld_id,
                             lg_konsDat, lg_konsEnde,
                             lg_txt
                             )
                 VALUES     (_aknr, _nort, _nchnr, _menge,
                             _dim1, _dim2, _dim3,
                             _lagzudat, _oldlgwen, _oldldid,
                             _oldKonsDat, _oldKonsEnde,
                             _oldlgtxt
                             )
              RETURNING lg_id
                   INTO _nlgid;
        ELSE
            _check_bezug := true;
        END IF;

        -- Umbuchungs-ID setzen
        UPDATE lagerlog SET lo_umbuchid = _umbuchid WHERE lo_id = (SELECT max(lo_id) FROM lagerlog);

        IF _check_bezug THEN
            IF _updlgwen IS DISTINCT FROM _oldlgwen AND _updlgwen IS NOT NULL THEN
                UPDATE lag SET lg_w_wen = NULL WHERE lg_id = _nlgid;
                PERFORM PRODAT_HINT(langtext(21283)); -- Zusammenlegen von Lagerbeständen hat Wareneingangsbezug entfernt
            END IF;
            IF _updldid IS DISTINCT FROM _oldldid AND _updldid IS NOT NULL THEN
                UPDATE lag SET lg_ld_id = NULL WHERE lg_id = _nlgid;
                PERFORM PRODAT_HINT(langtext(21284)); -- Zusammenlegen von Lagerbeständen hat Bestellbezug entfernt
            END IF;
        END IF;

        -- Seriennummern autom. umbuchen
        IF _autosernrwechsel THEN
           _umbuchsernr := '';
            FOR _r IN SELECT lgs_id, lgs_sernr
                        FROM lagsernr
                       WHERE lgs_lg_id = _olgid
                       ORDER BY insert_date, lgs_w_wen, lgs_sernr
                       LIMIT round(_menge)
            LOOP
                UPDATE lagsernr SET lgs_lg_id = _nlgid WHERE lgs_id = _r.lgs_id; -- Umbuchen der Seriennummer auf den neuen Lagerort
                _umbuchsernr := _umbuchsernr || E'\n' || _r.lgs_sernr;
            END LOOP;

            UPDATE lagerlog SET lo_txt = trim(_umbuchsernr) WHERE lo_umbuchid = _umbuchid;
        END IF;

        IF _dellag THEN
           DELETE FROM lag WHERE lg_id = _olgid AND lg_anztot <= 0;
        END IF;

        PERFORM TSystem.Settings__Set('umbuch', 'f');

        UPDATE lag SET lg_id = lg_id WHERE lg_id = _olgid AND lg_anztot <= 0; -- Alten Lagerort vom Trigger prüfen lassen, ob dieser gelöscht werden kann (auto_del_empty_lag).

        IF (_oort <> _nort) AND EXISTS(SELECT true FROM haenel WHERE _oort ILIKE hl_lg_ort) THEN -- Abgang vom HÄNEL-LIFT!
            SELECT ak_nrhl INTO _hnnr FROM art WHERE ak_nr = _aknr;
            INSERT INTO haenel_aktion (ha_aktion, ha_aknr, ha_nr_pos, ha_stk, ha_lgort) VALUES ('-', _hnnr, 'CHANGE', _menge, _oort);
        END IF;

        IF (_oort <> _nort) AND EXISTS(SELECT true FROM haenel WHERE _nort ILIKE hl_lg_ort) THEN -- dies geht auch an den HÄNEL-LIFT!
            SELECT ak_nrhl INTO _hnnr FROM art WHERE ak_nr = _aknr;
            INSERT INTO haenel_aktion (ha_aktion, ha_aknr, ha_nr_pos, ha_stk, ha_lgort) VALUES ('+', _hnnr, 'CHANGE', _menge, _nort);
        END IF;

        PERFORM tartikel.bestand_abgleich_intern(_aknr); -- wegen Umbuchen von Beistellmaterial usw.

        RETURN _nlgid;
    END $$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION z_99_deprecated.lagerortwechsel(
    _aknr    varchar, -- Artikel
    _oort    varchar, -- alter Lagerort
    _ochnr   varchar, -- alte ChargenNr.
    _nort    varchar, -- neuer Lagerort
    _nchnr   varchar, -- neue ChargenNr.
    _menge   numeric, -- Mengen
    _dim1    integer, -- Dimension
    _dim2    integer,
    _dim3    integer,
    -- Optionen
    _dellag            boolean,                   -- alten Lagerort bei Menge 0 löschen
    _autosernrwechsel  boolean   DEFAULT true, -- Seriennummern autom. mit Umbuchen.
    _umbuchid          integer   DEFAULT nextval('lag_umbuchid'),  -- Umbuchungs-ID
    _bemerk            text      DEFAULT null  -- Bemerkungstext aus Oberfläche
    )
    RETURNS integer
    AS $$
        SELECT tlager.lag__ort_chnr__lagerortwechsel(_aknr,
                                                     _oort,
                                                     _ochnr,
                                                     _nort,
                                                     _nchnr,
                                                     _menge,
                                                     _dim1,
                                                     _dim2,
                                                     _dim3,
                                                     -- Optionen
                                                     _dellag,
                                                     _autosernrwechsel,
                                                     _umbuchid,
                                                     _bemerk
                                                    );
    $$ LANGUAGE sql;
--

CREATE OR REPLACE FUNCTION z_99_deprecated.lagerortwechsel(
    _aknr    varchar, -- Artikel
    _oort    varchar, -- alter Lagerort
    _ochnr   varchar, -- alte ChargenNr.
    _nort    varchar, -- neuer Lagerort
    _nchnr   varchar, -- neue ChargenNr.
    _menge   numeric, -- Mengen
    _dim1    integer, -- Dimension
    _dim2    integer,
    _dim3    integer,
    -- Optionen
    _dellag            boolean,                   -- alten Lagerort bei Menge 0 löschen
    _autosernrwechsel  boolean   DEFAULT true, -- Seriennummern autom. mit Umbuchen.
    _umbuchid          integer   DEFAULT nextval('lag_umbuchid'),  -- Umbuchungs-ID
    _bemerk            text      DEFAULT null  -- Bemerkungstext aus Oberfläche
    )
    RETURNS integer
    AS $$
        SELECT tlager.lag__ort_chnr__lagerortwechsel(_aknr,
                                                     _oort,
                                                     _ochnr,
                                                     _nort,
                                                     _nchnr,
                                                     _menge,
                                                     _dim1,
                                                     _dim2,
                                                     _dim3,
                                                     -- Optionen
                                                     _dellag,
                                                     _autosernrwechsel,
                                                     _umbuchid,
                                                     _bemerk
                                                    );
    $$ LANGUAGE sql;

-- Lagerote austauschen (RTF 258)
-- TODO: Zusammenführung von: Konservierungsdatum, WE-Bezug, Bestell-Bezug
CREATE OR REPLACE FUNCTION lagerortaustausch(old_lg_ort VARCHAR, new_lg_ort VARCHAR) RETURNS VOID AS $$
  DECLARE lag_old RECORD;
          new_lg_id INTEGER;
  BEGIN
    IF old_lg_ort = new_lg_ort THEN
        RETURN;
    END IF;

    -- Lagerorte einfach umschreiben, wenn Ziellagerort, Charge nicht vorhanden.
    UPDATE lag
       SET lg_ort = new_lg_ort
    WHERE lg_ort = old_lg_ort
      AND NOT EXISTS(SELECT true FROM lag AS lagnew WHERE lagnew.lg_aknr = lag.lg_aknr AND lagnew.lg_chnr = lag.lg_chnr AND lagnew.lg_ort = new_lg_ort);
    --

    -- Lagerbestand zusammenführen, wenn Ziellagerort, Charge vorhanden (Rest, der nicht mit UPDATE umgeschrieben wurde).
    FOR lag_old IN (SELECT lg_id, lg_aknr, lg_chnr, lg_anztot FROM lag WHERE lg_ort = old_lg_ort) LOOP
        UPDATE lag SET
          lg_anztot= lg_anztot + lag_old.lg_anztot
        WHERE lg_ort = new_lg_ort
          AND lg_aknr = lag_old.lg_aknr
          AND lg_chnr = lag_old.lg_chnr
        RETURNING lg_id INTO new_lg_id;
        --

        -- Seriennummern mitziehen
        UPDATE lagsernr
           SET lgs_lg_id = new_lg_id
        WHERE lgs_lg_id = lag_old.lg_id;
        --
    END LOOP;

    -- Alte Lagerorte löschen.
    DELETE FROM lag WHERE lg_ort = old_lg_ort;

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- WARENEINGANGSSCHEIN ---------------------------------------------------------------------------------

CREATE TABLE wendat (
  w_wen                serial PRIMARY KEY,
  w_l_krz              varchar(21) REFERENCES adk ON UPDATE CASCADE, -- Kürzel des Lieferanten
  w_aknr               varchar(40) NOT NULL REFERENCES art ON UPDATE CASCADE, -- Gelieferter Artikel
  w_adpreis            numeric(20,8),
  w_lds_id             integer REFERENCES ldsdok,              -- Bestellung die Wareneingang auslöste
  w_ab_ix              integer,                                -- REFERENCES abk,
  w_zug_dat            timestamp without time zone DEFAULT currenttime(),   -- Bestätigtes Datum / Zugangs-Datum
  w_konsDat            date,                                   -- Konservierungsdatum (lg_konsDat)
  w_konsEnde           date,                                   -- Ablauf der Konservierung (lg_konsEnde)
  w_gewicht            numeric(12,4),                          -- Gewicht in kg
  w_lgort              varchar(50) NOT NULL DEFAULT '',        -- Wo Artikel eingelagert wurde
  w_lagort             varchar(50) NOT NULL DEFAULT '',        -- Lagerplatz
  w_lagbereich         varchar(50),                            -- Lagerbereich iVm TABLE lagerbereich
  w_lgchnr             varchar(50) NOT NULL DEFAULT '',        -- Chargennummer
  w_dim1               numeric(12,4) NOT NULL DEFAULT 0,
  w_dim2               numeric(12,4) NOT NULL DEFAULT 0,
  w_dim3               numeric(12,4) NOT NULL DEFAULT 0,
  w_zugang             numeric(14,6) NOT NULL CONSTRAINT xtt5111__w_zugang CHECK (w_zugang>=0), -- Eingelagerte Menge
  w_zugang_uf1         numeric(20,8),
  w_zug_mec            integer NOT NULL REFERENCES artmgc, -- Verwendete Mengeneinheit
  w_auss               numeric(14,6) CONSTRAINT xtt16454__w_auss CHECK (w_auss>=0),  -- Ausschuss bei Auswärts
  w_auss_uf1           numeric(20,8),
  w_auss_asg_id        integer, -- REFERENCES bdea_ausschussgruende ON UPDATE CASCADE -- Ausschussgrund
  w_adpreis_berech     bool DEFAULT FALSE,
  w_preis_inlief       bool DEFAULT FALSE,
  w_akhest             numeric(12,4),
  w_akvkpbas           numeric(12,4),
  w_lfsnr              varchar(30),                            -- Lieferscheinnummer Lieferant
  w_lbw                smallint,                               -- Lieferantenbewertung
  w_lbt                text,
  w_lbt_rtf            text,
  w_rech_eing          bool DEFAULT FALSE,                     -- Kennzeichen "Rechung erfasst / entfällt" damit Wareneingang "ausblendbar" bei Suche für Eingangsrechnung
  w_ks                 varchar(9) REFERENCES ksv,
  w_stkf               numeric(12,4) NOT NULL DEFAULT 0,       -- Menge die in Eingangsrechnung übernommen wurde (In Grundmengeneinheit)
  w_q_nr               integer, --REFERENCES QAB               -- QAB/Servicevorfall Nummer für den der Zugang erfolgte (Kunde schickt defekte Ware zurück, die frei eingebucht wird)
  w_l_nr               integer, --REFERENCES lifsch            -- RetoureLieferung : Rücklieferung auf eine Auslieferung (Verringerung) - http://redmine.prodat-sql.de/issues/6327
                                                                 -- Dieser Wareneingang ist eine RetoureLieferung auf einen Warenausgang
                                                                 -- ACHTUNG: BEACHTE l_w_wen: RetoureLieferung/Reklamationsauslieferung
                                                                 -- ODER: bei FolgeABK automatischer Lagerabgang nach Lagerzugang - https://redmine.prodat-sql.de/issues/9096
  w_menge_ok           bool,                                   -- Menge innerhalb der Toleranz
  w_a2_id              integer                                 -- Referenz zum letzten abgeschlossenen Arbeitsgang bei einem Halbfabrikat
                                                               -- ACHTUNG: könnte man prinzipiell auch für Auswärtsbearbeitungen verwenden, ist es aber aktuell nicht! AW wird über JOIN auf ldsdok und dort ld_a2_id erkannt!
);

SELECT setval('wendat_w_wen_seq', 10000);

-- Indizes
    CREATE INDEX wendat_lds_id ON wendat(w_lds_id) WHERE w_lds_id IS NOT NULL;
    CREATE INDEX wendat_w_aknr ON wendat(w_aknr, w_zug_dat);
    CREATE INDEX wendat_w_aknr_like ON wendat(w_aknr varchar_pattern_ops);
    CREATE INDEX wendat_w_lgort ON wendat(w_lgort);
    CREATE INDEX wendat_w_lgort_like ON wendat(w_lgort varchar_pattern_ops);
    CREATE INDEX wendat_w_lgchnr ON wendat(w_lgchnr);
    CREATE INDEX wendat_w_lgchnr_like ON wendat(w_lgchnr varchar_pattern_ops);
    CREATE INDEX wendat_w_lfsnr ON wendat(w_lfsnr) WHERE w_lfsnr IS NOT NULL;
    CREATE INDEX wendat_w_lfsnr_like ON wendat(AEOEUE_UPPER(w_lfsnr) varchar_pattern_ops) WHERE w_lfsnr IS NOT NULL;
    CREATE INDEX wendat_w_zud_dat ON wendat(CAST(w_zug_dat AS DATE));
    CREATE INDEX wendat_w_l_krz ON wendat(w_l_krz);
    CREATE INDEX wendat_w_l_krz_like ON wendat(w_l_krz varchar_pattern_ops);
    CREATE INDEX wendat_w_q_nr ON wendat(w_q_nr) WHERE w_q_nr IS NOT NULL;
    CREATE INDEX wendat_w_l_nr ON wendat(w_l_nr) WHERE w_l_nr IS NOT NULL;
    CREATE INDEX wendat_w_ab_ix ON wendat(w_ab_ix) WHERE w_ab_ix IS NOT NULL;
    CREATE INDEX wendat_w_a2_id ON wendat(w_a2_id) WHERE w_a2_id IS NOT NULL;
--

--Bei Erstellen oder Aktualisierung eines Lagerzugangs, aktualisieren wir w_lgort > Lagerort aus Lagerort und Lagerbereich  #6259
CREATE OR REPLACE FUNCTION wendat__b_10_iu_lagort_lgort() RETURNS TRIGGER AS $$
  BEGIN
    new.w_lgort := coalesce(NullIf(TLager.lag__lagort__bereich__generate(new.w_lagbereich, new.w_lagort)
                                   , ''
                                  )
                            , new.w_lgort
                            );--COALESCE: falls in alten Oberflächen doch nur in das w_lgort geschrieben wird.
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__b_10_iu_lagort_lgort
    BEFORE INSERT OR UPDATE OF w_lagort, w_lagbereich
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__b_10_iu_lagort_lgort();
--

--
CREATE OR REPLACE FUNCTION wendat__b_20_iu_lgort_lagort() RETURNS TRIGGER AS $$
  DECLARE r RECORD;
  BEGIN
    SELECT * INTO r FROM TLager.lag__lagort__bereich__split(new.w_lgort);
    new.w_lgort:=r.lgort;
    new.w_lagort:=r.lagort;
    new.w_lagbereich:=r.lagbereich;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__b_20_iu_lgort_lagort
    BEFORE INSERT OR UPDATE OF w_lgort
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__b_20_iu_lgort_lagort();
--

CREATE OR REPLACE FUNCTION wendat__b_i() RETURNS TRIGGER AS $$
  DECLARE
        _code      varchar;
        _ld_qnr    integer;
  BEGIN

        -- Statistikdaten aus Art rein.
        SELECT ak_hest, ak_vkpbas
          INTO new.w_akhest, new.w_akvkpbas
          FROM art
         WHERE ak_nr = new.w_aknr;

        -- Retourelieferung: http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Lager   ... kein Bezug, da sonst ld_stkl falsch
        _code := NULL;

        -- bestellbezogen?, Dann Defaults aus Bestellung was nicht mitgegeben wurde
        IF new.w_lds_id IS NOT NULL THEN
           SELECT ld_code,
                  ld_ks,
                  -- Defaults aus Bestellung
                  coalesce(new.w_l_krz, ld_kn),
                  coalesce(new.w_aknr, ld_aknr),
                  coalesce(new.w_zug_mec, ld_mce),
                  -- coalesce(new.w_zugang, ld_stk), -- NICHT. Die Menge MUSS eingegeben werden, zu hohe Fehlerquelle wenn in Oberfläche einfach speichern gedrückt wird.
                  coalesce(new.w_q_nr, ld_q_nr)
             INTO _code,
                  new.w_ks,
                  new.w_l_krz,
                  new.w_aknr,
                  new.w_zug_mec,
                  -- new.w_zugang,
                  new.w_q_nr
             FROM ldsdok
             LEFT JOIN ldsdokdokutxt ON (ltd_dokunr = ld_dokunr)
            WHERE ld_id = new.w_lds_id;
        ELSE
           IF new.w_zug_mec IS NULL THEN
              new.w_zug_mec := tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( new.w_aknr );
            END IF;
        END IF;

        -- Interner Lagerzugang bzw. vom Kunden zurückgeschickte Ware erfordert keine Rechnung
        IF    (_code = 'I')
           OR (_code IS NULL AND new.w_q_nr IS NOT NULL)
        THEN
               new.w_rech_eing := true;
        END IF;

        -- Bewertung Lagerzugang erforderlich (wenn Kürzel Lieferant da ist)
        IF     _code = 'E'
           AND new.w_l_krz IS NOT NULL
           AND new.w_lbw IS NULL
           AND TSystem.Settings__GetBool('BEWERLAGZU')
        THEN
            RAISE EXCEPTION 'null value in not-null "w_lbw"';
        END IF;

        RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__b_i
    BEFORE INSERT
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__b_i();

-- GME umrechnen, Lieferantenbewertung sichersstellen, w_q_nr aus Bestellung setzen
CREATE OR REPLACE FUNCTION wendat__b_iu() RETURNS TRIGGER AS $$
  DECLARE _uf       numeric;
  BEGIN

    -- Der Datensatz wurde mit einer für den Artikel ungültigen Mengeneinheit gespeichert. Tabelle: %, ID: %, Feld: %, ART-Nr.: %, Artmgc-ID: %
    IF new.w_aknr IS NOT NULL AND NOT TArtikel.me__art__artmgc__m_ids__valid(new.w_aknr, new.w_zug_mec)
    THEN
       RAISE EXCEPTION '%', Format(lang_text(13795), 'wendat',    new.w_wen::varchar,
                                                     'w_zug_mec', new.w_aknr,
                                                                  new.w_zug_mec::varchar
                                  );
    END IF;

    new.w_lgort  := upper( trim(both ' ' FROM new.w_lgort) );
    new.w_lgchnr := upper( trim(both ' ' FROM new.w_lgchnr) );

    _uf := m_uf FROM artmgc WHERE m_id = new.w_zug_mec;

    new.w_zugang_uf1 := new.w_zugang / _uf;
    new.w_auss_uf1   := new.w_auss   / _uf;

    IF (trim(new.w_lfsnr) = '') THEN new.w_lfsnr := NULL; END IF;
    IF (trim(new.w_ks)    = '') THEN new.w_ks    := NULL; END IF;

    -- abk-index
    IF new.w_lds_id IS NOT NULL AND new.w_ab_ix IS NULL THEN
        new.w_ab_ix:=ld_abk FROM ldsdok WHERE ld_id=new.w_lds_id;
    END IF;
    IF new.w_a2_id IS NOT NULL AND new.w_ab_ix IS NULL THEN
       new.w_ab_ix := a2_ab_ix FROM ab2 WHERE a2_id = new.w_a2_id;
    END IF;

    -- Lagerbewegung UE muss vom Insert festegelegt sein und darf nicht geändert werden (Lagerlog Einträge usw)
    -- es darf GAR NICHTS geändert werden. Daher wird hier nur geprüft ob w_a2_id gesetzt ist (war) dann ist ein Update generell nicht möglich
    -- ob es hier evtl (bei internen updates) nochmal nachjustiert werden muss, wird sich zeigen
    IF     tg_op = 'UPDATE'
       AND (   (old.w_a2_id IS NOT NULL)
            OR (new.w_a2_id IS NOT NULL)
            )
    THEN
       RAISE EXCEPTION 'wendat ue cannot be modified xtt28819'; -- cannot change wendat ue to not ue'
    END IF;

    -- Bei RetoureLieferung (#6327) Fallback des Konservierungsdatum auf heute, wenn Artikel K-Pflicht hat.
    IF      new.w_l_nr IS NOT NULL
       AND (new.w_konsDat IS NULL OR new.w_konsEnde IS NULL)
    THEN
        IF EXISTS(SELECT true FROM art WHERE ak_nr = new.w_aknr AND ak_konsreq) THEN
            new.w_konsDat  := coalesce(new.w_konsDat, current_date);
            new.w_konsEnde := coalesce(new.w_konsEnde, current_date);
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;
--
  CREATE TRIGGER wendat__b_iu
    BEFORE UPDATE OR INSERT
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__b_iu();
--

-- Verfügbarkeit in Art und Lag, Liefermenge Bestellung, Lagerort anlegen, updaten, sperren, QAB anlegen, Lagerlog
CREATE OR REPLACE FUNCTION wendat__a_10_i() RETURNS TRIGGER AS $$
  DECLARE hnnr            varchar; --hänel-artikelnummer
          sperr           boolean;
          doqab           boolean;
          _lag            lag;
          rck_w_wen       integer; --w_wen: WareneingangsNummer. Variable: Rückbuchung. Bei Rückbuchung auf einen alten Lagerabgang wird die Ursprungs-WE-Nr (also die des Lieferantenlagerzugangs) wieder gesetzt. > http://redmine.prodat-sql.de/issues/6911
          rck_w_lds_id    integer;
          lgwwen          integer;
          lgldid          integer;
          _isuebuchung    boolean;
          _isauswaerts    boolean;
  BEGIN
    IF current_user='syncro' THEN
        RETURN new;
    END IF;
    --wiederherstellen der ursprünglichen bezüge bei RetoureLieferung; http://redmine.prodat-sql.de/issues/6911
    IF new.w_l_nr IS NOT NULL THEN
        SELECT l_w_wen, l_ld_id INTO rck_w_wen, rck_w_lds_id FROM lifsch WHERE l_nr=new.w_l_nr; --Wareneingang des Ursprünglichen Lagerorts
    END IF;

    -- Lagerzugang UE mit Arbeitsgang
    -- -- ACHTUNG: könnte man prinzipiell auch für Auswärtsbearbeitungen verwenden, ist es aber aktuell nicht! AW wird über JOIN auf ldsdok und dort ld_a2_id erkannt!
    _isuebuchung := new.w_a2_id IS NOT NULL;

    IF         new.w_lds_id IS NOT NULL
               -- Lagerzugang UE: Zugang auf Arbeitsgang führt nicht zur Änderung Menge geliefert #16658
       AND NOT _isuebuchung
    THEN

        PERFORM disablemodified();

        -- Menge geliefert in Bestellung anpassen
        UPDATE ldsdok
           SET ld_stkl   = ld_stkl + new.w_zugang_uf1,
               -- Kennzeichen für Wareneingang ohne Zugangsbuchung. Wir wissen evtl nicht in welchem Statusfeld das steht
               -- Zurücksetzen wenn Zubuchung geschehen (somit kann das Kennzeichen für eine weitere Teillieferung wieder gesetzt werden.)
               ld_bstat  = TSystem.Enum_DelValue( ld_bstat,  'W' ),
               ld_bstat1 = TSystem.Enum_DelValue( ld_bstat1, 'W' ),
               ld_bstat2 = TSystem.Enum_DelValue( ld_bstat2, 'W' )
         WHERE ld_id = new.w_lds_id;

        PERFORM enablemodified();
    END IF;

    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'F'); -- Lagerlogging abschalten, wird hier in dem Trigger geführt
    -- Bei Lagerbewertung eventuell Artikel sperren und / oder QAB erstellen, wenn die Bewertung das erfordert.
    IF new.w_lbw IS NOT NULL THEN
        SELECT lb_sperr, lb_makeqab INTO sperr, doqab FROM lbw WHERE lb_note = new.w_lbw;
    END IF;

    -- QAB anlegen wenn erforderlich. Muss vor INSERT INTO lag ... passieren da dort WEB-Anlage prüft ob QAB vorhanden
    IF new.w_lbw IS NOT NULL AND doQab THEN
        INSERT INTO qab (q_nr, q_w_wen,q_ak_nr,q_dat,q_festext, q_typ, q_isService,q_stk, q_mecode,q_referenz)
        VALUES ( getnumcirclenr('qab')::INTEGER, new.w_wen, new.w_aknr,Current_Date, FALSE, 3,FALSE, new.w_zugang,
                 (SELECT m_mgcode FROM artmgc WHERE m_id = new.w_zug_mec ORDER BY m_mgcode LIMIT 1), -- Mengeneinheit aus Wareneingang mitnehmen
                 (SELECT ld_ekref FROM ldsdok WHERE new.w_lds_id = ld_id) -- Bestellreferenz als QAB-Referenz
               );
        IF found THEN PERFORM prodat_hint( 'neuer qab xtt13073' ); END IF; -- Hinweis: Neuer QAB erstellt
    END IF;

    /******************************** HIER IST SCHLUSS, WENN AUSWAERTSRUECKBUCHUNG ODER ARTIKEL NICHT LAGERGEFÜHRT ODER ARBEITSPAKET ********************/

    _isauswaerts := coalesce((SELECT true FROM ldsdok WHERE ld_id = new.w_lds_id AND ld_a2_id IS NOT NULL), False);

    IF    (NOT TArtikel.art__ac_i__is__50(new.w_aknr))
       OR _isauswaerts
    THEN
        PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'T'); -- Lagerlogging anschalten

        IF NOT _isauswaerts THEN
           PERFORM prodat_hint( 'wendat auf dienstleistung xtt28820'); -- Artikel ist nicht lagergeführt oder ein Arbeitspaket
        END IF;

        RETURN new;
    END IF;

    -- Lagersatz anlegen oder Updaten. Beim anlegen wird u.a. geprueft ob WEB angelegt wird.
    _lag := TLager.lag__from__wendat(new);

    -- Lagerort noch nicht vorhanden
    IF _lag.lg_id IS NULL THEN
       INSERT INTO lag
                   (lg_aknr,
                    lg_ort, lg_chnr, lg_anztot,
                    lg_dim1, lg_dim2, lg_dim3,
                    lg_w_wen, lg_ld_id,
                    lg_konsDat, lg_konsEnde,
                    lg_txt
                    )
            VALUES (new.w_aknr,
                    new.w_lgort, new.w_lgchnr, new.w_zugang_uf1,
                    new.w_dim1, new.w_dim2, new.w_dim3,
                    COALESCE(rck_w_wen, new.w_wen), COALESCE(rck_w_lds_id, new.w_lds_id),
                    new.w_konsDat, new.w_konsEnde,
                    new.w_lbt
                    );
    ELSE
        IF _isuebuchung AND _lag.lg_anztot > 0 THEN
           RAISE EXCEPTION 'Lagerzugang UE auf existierenden Lagerort+Charge nicht möglich (Artikel können nicht gemischt werden): %, %', _lag.lg_ort, _lag.lg_chnr;
        END IF;

        UPDATE lag SET lg_anztot   = lg_anztot + new.w_zugang_uf1,
                                     -- lagerort war bereits gesperrt (zB Lagerzugang UE)  >  dann bleibt es natürlich und wird nicht durch die Grundeinstellung entsperrt
                       lg_sperr    = lg_sperr OR lgort_dosperr(new.w_lgort),
                       lg_w_wen    = CASE WHEN lg_anztot<=0 THEN COALESCE(rck_w_wen, new.w_wen) ELSE lg_w_wen END, -- Bezug erneut herstellen bei Bestand <=0 (Standardlagerort)
                       lg_ld_id    = CASE WHEN lg_anztot<=0 THEN COALESCE(rck_w_lds_id, new.w_lds_id) ELSE lg_ld_id END, -- ebenso
                       lg_konsdat  = CASE WHEN lg_anztot<=0 THEN new.w_konsdat ELSE LEAST(lg_konsdat, new.w_konsdat) END,
                       lg_konsEnde = CASE WHEN lg_anztot<=0 THEN new.w_konsende ELSE LEAST(lg_konsende, new.w_konsende) END,
                       lg_txt      = CASE WHEN lg_txt IS NULL THEN new.w_lbt ELSE lg_txt || COALESCE(E'\n\n' || new.w_lbt, '') END
                 WHERE lg_id = _lag.lg_id
             RETURNING lg_w_wen, lg_ld_id
                  INTO lgwwen, lgldid;

        IF lgwwen IS DISTINCT FROM COALESCE(rck_w_wen, new.w_wen) AND lgwwen IS NOT NULL THEN
            UPDATE lag SET lg_w_wen = NULL WHERE lg_id = _lag.lg_id;
            PERFORM PRODAT_HINT(langtext(21283)); -- Zusammenlegen von Lagerbeständen hat Wareneingangsbezug entfernt
        END IF;
        IF lgldid IS DISTINCT FROM COALESCE(rck_w_lds_id, new.w_lds_id) AND lgldid IS NOT NULL THEN
            UPDATE lag SET lg_ld_id = NULL WHERE lg_id = _lag.lg_id;
            PERFORM PRODAT_HINT(langtext(21284)); -- Zusammenlegen von Lagerbeständen hat Bestellbezug entfernt
        END IF;
    END IF;

    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'T'); -- Lagerlogging anschalten

    PERFORM TLager.lagerlog__create('+', new); --- #14796 wendat__a_10_i()

    IF EXISTS(SELECT true FROM haenel WHERE new.w_lgort ILIKE hl_lg_ort) THEN -- dies geht auch an den HÄNEL-LIFT!!!
        SELECT ak_nrhl INTO hnnr FROM art WHERE ak_nr=new.w_aknr;
        INSERT INTO haenel_aktion (ha_aktion, ha_aknr, ha_nr_pos, ha_stk, ha_lgort) VALUES ('+', hnnr, new.w_wen, new.w_zugang_uf1, new.w_lgort);
    END IF;
    -- Lagerort noch sperren wenn Bewertung das verlangt
    IF new.w_lbw IS NOT NULL THEN
        IF sperr=TRUE THEN
            UPDATE lag SET lg_sperr = TRUE WHERE lg_aknr = new.w_aknr AND lg_ort = new.w_lgort AND lg_chnr = new.w_lgchnr;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_10_i
    AFTER INSERT
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_10_i();
--

-- Seriennummern in LagSerNr vorbereiten
CREATE OR REPLACE FUNCTION wendat__a_12_i__seriennummer() RETURNS TRIGGER AS $$
  DECLARE
    lgid integer;
    ser_cnt integer;
    use_sn_nr boolean;
    r_count integer;
  BEGIN
    use_sn_nr :=    -- Sind Vorgabe-Seriennummern vorhanden?
                    EXISTS(SELECT * FROM TLager.mapsernr__ms_ids__by__ld_id__get(new.w_lds_id) LIMIT 1)
                 OR
                    -- Oder besteht Seriennummernpflicht?
                    ak_sernrreq FROM art WHERE ak_nr = new.w_aknr;

    IF use_sn_nr THEN --Seriennummern verwenden/erzeugen?
        lgid := TLager.lag__lg_id__from__wendat(new);

        IF lgid IS NULL THEN
           -- wir sind im after insert. hier muss es eigentlich einen lag-datensatz geben! außer: new.w_zugang = 0  was man evtl noch klären müßte
           RAISE EXCEPTION 'wendat__a_12_i__seriennummer lag not found';
        END IF;

        -- Beim 'Lagerzugang frei' als Ausnahme falls eine QAB/SV-Nummer im WE hinterlegt ist,
        -- erst Seriennummern von Servicevorfall einfügen
        IF    new.w_q_nr IS NOT NULL --QAB/SV-Nummer im WE hinterlegt
          AND new.w_lds_id IS NULL   --Lagerzugang bestellbezogen ausschliessen
          AND (SELECT q_sernr FROM qab WHERE q_nr = new.w_q_nr LIMIT 1) IS NOT NULL
        THEN
          WITH
            Vorgabe_Seriennummern_SV AS (
              SELECT
                sub AS q_sernr
              FROM
                qab
                JOIN LATERAL unnest(string_to_array(TRIM(q_sernr), E'\r\n', '')) AS sub ON true
              WHERE
                qab.q_nr = new.w_q_nr
                AND sub IS DISTINCT FROM NULL),

            Gebuchte_Seriennummern_SV AS (
              SELECT
                wendat.w_wen,
                lagsernr.lgs_sernr
              FROM
                wendat
                JOIN lagsernr ON lagsernr.lgs_w_wen = wendat.w_wen
              WHERE
                wendat.w_q_nr = new.w_q_nr)

            INSERT INTO lagsernr
                       (lgs_w_wen, lgs_lg_id, lgs_sernr)
                 SELECT new.w_wen, lgid,      q_sernr
                 FROM
                   Vorgabe_Seriennummern_SV
                   LEFT JOIN Gebuchte_Seriennummern_SV ON lgs_sernr = q_sernr
                 WHERE
                   w_wen IS NULL
                   OR w_wen = new.w_wen;
            --
            GET DIAGNOSTICS r_count = ROW_COUNT; --Anzahl der eingetragenen Datensätze
        ELSE
          r_count := 0;
        END IF;

        -- Falls es Teilbuchungen gab, wird geprüft ob, durch vorherige Lagerzugänge eine Seriennummer aus den Vorgaben bereits zugewiesen wurde,
        --   das heißt, die in der Bestellung vorgegebene Seriennummer ist mit einem Wareneingang verknpüft.
        -- Somit gab es zB Seriennummer A,B,C,D und Seriennummer C war bereits durch einen vorherigen Wareneingang verknüpft wurden.

        INSERT INTO lagsernr
                    (lgs_w_wen, lgs_lg_id, lgs_sernr)
              SELECT new.w_wen, lgid, sub.sernr
              FROM (
                     SELECT lagsernr__presetting__create AS sernr
                     FROM TLager.lagsernr__presetting__create( new.w_lds_id, (new.w_zugang - r_count)::integer, new.w_wen )
                     WHERE
                        NOT EXISTS ( -- Seriennummer bereits zugeordnet
                                    SELECT 1 FROM lagsernr
                                    JOIN wendat ON w_wen = lgs_w_wen AND w_lds_id = new.w_lds_id
                                    WHERE lgs_sernr = lagsernr__presetting__create
                                )
                        AND NOT EXISTS ( -- ist Vorgabeseriennummer
                                    SELECT 1 FROM mapsernr
                                    JOIN lagsernr ON ms_lgs_id = lgs_id AND lgs_sernr = lagsernr__presetting__create
                                    WHERE ms_pkey IS NOT null AND ms_table = 'ldsdok'::REGCLASS
                                )
                     LIMIT new.w_zugang
                   ) AS sub
        ;


        -- Zugeordnete SN Zählen
        SELECT Count(true)
        INTO ser_cnt
        FROM lagsernr
        WHERE lgs_w_wen = new.w_wen;

        -- Automatische Zuordnung der Vorgabeseriennummern aufsteigend
        IF ser_cnt < new.w_zugang THEN
          UPDATE lagsernr
          SET lgs_w_wen = new.w_wen
          WHERE lgs_id IN (SELECT lgs_id
                           FROM lagsernr
                           JOIN mapsernr ON ms_lgs_id = lgs_id AND ms_pkey::INTEGER = new.w_lds_id AND ms_table = 'ldsdok'::REGCLASS
                           WHERE lgs_w_wen IS NULL
                           ORDER BY lgs_sernr ASC
                           LIMIT (new.w_zugang - ser_cnt));
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_12_i__seriennummer
    AFTER INSERT
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_12_i__seriennummer();
--

-- Seriennummern in LagSerNr nachziehen, wenn Lagerort des LZ geändert
CREATE OR REPLACE FUNCTION wendat__a_60_u__seriennummer() RETURNS TRIGGER AS $$
  DECLARE
    _lgid_old integer;
    _lgid_new integer;
  BEGIN

    -- Dieser Trigger darf erst nach dem möglichen Anlegen des neuen Lagerorts ausgeführt werden,
    -- denn die ID des neuen Lagerorts soll der Seriennummer übergeben werden.

    IF    new.w_lgort  IS DISTINCT FROM old.w_lgort
       OR new.w_lgchnr IS DISTINCT FROM old.w_lgchnr
    THEN

      _lgid_old := (TLager.lag__from__wendat(old)).lg_id;
      _lgid_new := (TLager.lag__from__wendat(new)).lg_id;

      IF _lgid_old IS NOT null AND _lgid_new IS NOT null THEN
        UPDATE lagsernr SET lgs_lg_id = _lgid_new
        WHERE lgs_w_wen = old.w_wen AND lgs_lg_id = _lgid_old;
      END IF;

    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_60_u__seriennummer
    AFTER UPDATE OF w_lgort, w_lgchnr
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_60_u__seriennummer();
--

-- Konservierungsdatum
CREATE OR REPLACE FUNCTION wendat__a_12_iu__konservierung() RETURNS TRIGGER AS $$
  BEGIN
    IF ( SELECT ak_konsreq FROM art WHERE ak_nr = new.w_aknr ) AND ( new.w_konsDat IS null ) THEN
        -- Konservierungsdatum erforderlich [Artikel= | Lagerort= | Charge= ]
        RAISE EXCEPTION '%', FORMAT( '%s ' || E'\r\n' || '[%s=%s | %s=%s | %s=%s]', lang_text(17990), Lang_text(234),
          new.w_aknr, lang_text(509), new.w_lgort, lang_text(28003), new.w_lgchnr );
    END IF;

    UPDATE lag SET lg_konsDat = new.w_konsDat, lg_konsEnde = new.w_konsEnde
    WHERE lg_w_wen = new.w_wen  -- Lager anpassen, wenn WEB geändert wird (KonsDat in WEB-Bearbeiten deaktiviert, wenn bereits im Lager umgebucht wurde)
      AND lg_konsDat IS DISTINCT FROM new.w_konsDat    -- Trigger nicht auslösen, wenn sich nix ändert
      AND lg_konsEnde IS DISTINCT FROM new.w_konsEnde;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_12_iu__konservierung
    AFTER INSERT OR UPDATE OF w_aknr, w_konsDat, w_konsEnde
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_12_iu__konservierung();


-- Mengenänderungen, Lieferantenbewertungen nachziehen, Bestandswirksamkeit prüfen
CREATE OR REPLACE FUNCTION wendat__a_14_u() RETURNS TRIGGER AS $$
  DECLARE olgid                     INTEGER;
          R                         RECORD;
          oldmenge                  NUMERIC;
          wendatbestandswirksam     BOOL;
          oldDoQAB                  BOOL;
          DoQAB                     BOOL;
  BEGIN
    -- Die Lieferantenbewertung hat sich geändert. Prüfen ob wir was am QAB machen müssen.
    IF (new.w_lbw IS DISTINCT FROM old.w_lbw) THEN
        -- Erforderte die alte Lagerbewertung einen QAB?
        oldDoQab:=False;
        IF (old.w_lbw IS NOT NULL ) THEN
            SELECT lb_makeqab INTO oldDoQAB FROM lbw WHERE lb_note = old.w_lbw;
        END IF;

        -- Erfordert die neue Lagerbewertung einen QAB?
        doqab:=False;
        IF (new.w_lbw IS NOT NULL ) THEN
            SELECT lb_makeqab INTO doqab FROM lbw WHERE lb_note = new.w_lbw;
        END IF;
        --Neue Bewertung erfordert einen QAB --> Also einen anlegen, wenn es noch keinen (oder keinen offenen mehr) gibt.
        IF DoQAB THEN
            INSERT INTO qab (q_nr,q_w_wen,q_ak_nr,q_dat,q_festext, q_typ, q_isService,q_stk,q_mecode,q_referenz)
            SELECT getnumcirclenr('qab')::INTEGER, new.w_wen, new.w_aknr, Current_Date, FALSE, 3,FALSE, new.w_zugang,
                   (SELECT m_mgcode FROM artmgc WHERE m_id = new.w_zug_mec ORDER BY m_mgcode LIMIT 1), -- Mengeneinheit aus Wareneingang mitnehmen
                   (SELECT ld_ekref FROM ldsdok WHERE new.w_lds_id = ld_id) -- Bestellreferenz als QAB-Referenz
            WHERE NOT EXISTS(SELECT true FROM qab WHERE q_w_wen = new.w_wen AND NOT q_isService AND NOT q_def);
            IF FOUND THEN PERFORM Prodat_Hint(lang_text(13073)); END IF; --Hinweis: Neuer QAB erstellt
        END IF;
        --Neue Bewertung benötigt keinen QAB oder ist leer und die alte erforderte einen QAB, dann war der existierende automatisch angelegt und wir machen den zu.
        IF (Not DoQab) AND oldDoQAB THEN
            UPDATE qab SET q_def=true WHERE q_w_wen = new.w_wen AND NOT q_isService AND NOT q_def;
            IF FOUND THEN PERFORM Prodat_Hint(lang_text(13074)); END IF; --Hinweis: QAB geschlossen
        END IF;
    END IF;

    --Abfangen wenn kein Wechsel am Lager war!
    IF (current_user='syncro')OR--nutzer syncro oder keine Lageränderungen
        ((new.w_zugang_uf1=old.w_zugang_uf1)AND(old.w_aknr=new.w_aknr)AND(old.w_lgort=new.w_lgort)AND(old.w_lgchnr=new.w_lgchnr)AND(old.w_dim1=new.w_dim1)AND(old.w_dim2=new.w_dim2)AND(old.w_dim3=new.w_dim3)) THEN
        RETURN new;
    END IF;

    IF new.w_zugang_uf1<>old.w_zugang_uf1 OR new.w_lds_id<>old.w_lds_id THEN
        PERFORM disablemodified();
        UPDATE ldsdok SET ld_stkl=ld_stkl-old.w_zugang_uf1 WHERE ld_id=old.w_lds_id;
        UPDATE ldsdok SET ld_stkl=ld_stkl+new.w_zugang_uf1 WHERE ld_id=new.w_lds_id;
        PERFORM enablemodified();
    END IF;

    /******************************** HIER IST SCHLUSS, WENN AUSWAERTSRUECKBUCHUNG ODER ARTIKEL NICHT LAGERGEFÜHRT ODER ARBEITSPAKET ********************/
    IF (NOT TArtikel.art__ac_i__is__50(new.w_aknr)) THEN
        RETURN new;
    END IF;

    --
    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'F'); --Lagerlogging abschalten, wird hier in dem Trigger geführt

    olgid := TLager.lag__lg_id__from__wendat(old);

    IF new.w_lgort<>old.w_lgort OR new.w_lgchnr<>old.w_lgchnr THEN
        IF EXISTS(SELECT true FROM lagsernr WHERE lgs_lg_id=olgid AND lgs_l_nr IS NOT NULL) THEN
            RAISE EXCEPTION 'xtt4960 - cannot change wendat with sernr - %', new.w_lgort;
        END IF;
    END IF;

    wendatbestandswirksam := tlager.wendatchange_checkbestwirksam(new) OR old.w_zugang_uf1 < new.w_zugang_uf1;-- ... oder wir Erhöhen die Zugangsmenge, dann wird einfach die differenz zugebucht!;

    IF wendatbestandswirksam THEN
        -- neuen Lagerort um neuen Zugang erhöhen
        UPDATE lag
           SET lg_anztot = lg_anztot + new.w_zugang_uf1
         WHERE lg_id = TLager.lag__lg_id__from__wendat(new);

        IF NOT found THEN
            --Lagerort gibts nicht, dann anlegen!
            PERFORM TLager.lag__from__wendat__create(new);
        END IF;

        -- ist Abzug des alten Zugangs möglich, abhängig von Einstellung: Negative Lagerbestände erlauben
        IF     TSystem.Settings__GetBool('no_neg_lag')
           AND coalesce( ((TLager.lag__from__wendat(old)).lg_anztot - old.w_zugang_uf1), -1 ) < 0
        THEN
            PERFORM PRODAT_TEXT(lang_text(16134) || old.w_lgort); -- Achtung! Inventur nötig. Korrektur unterschreitet 0 bzw. Lagerort existiert nicht mehr (nachfolgende Buchungen vorhanden). Lagerort:
            wendatbestandswirksam:=FALSE; -- Damit in die Lagerlog ein ! kommt.
        END IF;

        -- alten Lagerort um alten Zugang verringern
        UPDATE lag
          SET lg_anztot=lg_anztot-old.w_zugang_uf1
         WHERE lg_id = TLager.lag__lg_id__from__wendat(old);

        IF NOT found AND NOT TSystem.Settings__GetBool('no_neg_lag') THEN
            --alter Lagerort nicht mehr vorhanden, dann anlegen!
            --nur wirksam wenn negative Bestände erlaubt, weil sonst sowieso unter 0
            INSERT INTO lag
                        (lg_aknr, lg_ort, lg_chnr,
                         lg_dim1, lg_dim2, lg_dim3,
                         lg_anztot, lg_w_wen, lg_ld_id,
                         lg_konsDat, lg_konsEnde,
                         lg_txt
                         )
                 VALUES ( old.w_aknr, old.w_lgort, old.w_lgchnr,
                          old.w_dim1, old.w_dim2, old.w_dim3,
                         -old.w_zugang_uf1, new.w_wen, new.w_lds_id,
                          new.w_konsDat, new.w_konsEnde,
                          new.w_lbt
                          );
        END IF;
    ELSE
        PERFORM PRODAT_TEXT(lang_text(16134) || new.w_lgort); -- Achtung! Inventur nötig. Korrektur unterschreitet 0 bzw. Lagerort existiert nicht mehr (nachfolgende Buchungen vorhanden). Lagerort:
    END IF;

    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'T');--Lagerlogging anschalten, wird hier in dem Trigger geführt

    PERFORM TLager.lagerlog__create('=+'||IFTHEN(wendatbestandswirksam, '', ' !'), new, old); --- #14796

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_14_u
    AFTER UPDATE
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_14_u();
--

-- Prüfen ob Beistell-Lagerort notwendig ist.
CREATE OR REPLACE FUNCTION wendat__a_iu_check_reklam_lgort() RETURNS TRIGGER AS $$
  DECLARE needsReklam     BOOLEAN;
          fertArt         BOOLEAN;
  BEGIN
    /* Bei einem Reklamationszugang durch den Kunde oder Lagerzugang auf Nachbearbeitung  muss auf einen Beistelllagerort
      gebucht werden, der nicht verfügbar oder werthaltig ist, da die Teile nicht uns gehören sondern dem Kunde.
      Wenn es keiner ist, muss der Lagerzugang unterbunden werden.*/

    needsReklam:=False;

    -- Bestellbezogen? ld_nbedarf = TRUE => Nicht bedarfswirksam => Nachbearbeitung, gehört auf einen Reklamationslagerort
    IF new.w_lds_id IS NOT NULL THEN
        needsReklam := EXISTS (SELECT true FROM ldsdok WHERE ld_id = new.w_lds_id AND ld_q_nr IS NOT NULL AND ld_nbedarf);
    ELSE
        /*Freier Lagerzugang ist schwieriger: Siehe http://redmine.prodat-sql.de/issues/5070  Ist das:
          # ein freier Lagerzugang mit QAB aus Lief.Bewertung ?  => Kein Fertigungsartikel
          # eine beanstandete Auswärtsbearbeitung    ?           => Muss Bestellbezug haben
          # Zubuchung nach einer Nachbearbeitung     ?           => Ohne Lagerbuchungen "Kurzer Weg" => Kein Lagerzugang
                                                                 => Mit Lagerbuchungen, benötigt Nachbearbeitungsauftrag => Bestellbezug
          # ein Reklamationszugang durch den Kunde   ?           => Müsste Fertigungsteil sein und nicht von '#' kommen
          Dafür gibt es bisher kein eindeutigeres Kriterium.     */

        fertArt     := ak_fertigung FROM art WHERE ak_nr = new.w_aknr;
        needsReklam := new.w_q_nr IS NOT NULL;  -- QAB-Bezug vorhanden
        needsReklam := needsReklam AND fertArt AND (new.w_l_krz <> '#'); -- Fertigungsartikel von außerhalb ohne Bestellbezug => Kundenreklamation

    END IF;

    -- Darf gebucht werden.
    IF NOT needsReklam THEN
        RETURN new;
    END IF;

    -- Standardlagerort gegenprüfen, der muss angelegt sein.
    IF NOT EXISTS(SELECT true FROM lagerorte__get_setup(new.w_lgort) WHERE _beistell AND NOT _verfgbar AND NOT _werthg) THEN
        RAISE EXCEPTION 'xtt13060'; -- Fehler: Wenn Teile vom Kunden reklamiert und zurückgeschickt wurden oder aus einer Nachbearbeitung stammen,
        -- müssen sie auf einem Beistell-Lagerort gebucht werden, der weder in die Verfügbarkeit, noch in die Werthaltigkeit eingeht.
    END IF;

    -- Wenn QAB bezogener Wareneingang ohne Bestellbezug, dann ist das die Kundenreklamation. Wir schreiben die Wareneingangsnummer in den QAB zurück.
    --  Das ist Voraussetzung für die Retourelieferung.
    IF new.w_q_nr IS NOT NULL AND new.w_lds_id IS NULL THEN
        UPDATE qab SET q_w_wen = new.w_wen WHERE q_nr = new.w_q_nr AND q_w_wen IS NULL;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_iu_check_reklam_lgort
    AFTER INSERT OR UPDATE
    ON wendat
    FOR EACH ROW
    WHEN ((new.w_lds_id IS NOT NULL) OR (new.w_q_nr IS NOT NULL))
    EXECUTE PROCEDURE wendat__a_iu_check_reklam_lgort();
--

-- Verknüpfung von Reklamations-Wareneingang und zugehörigem QAB löschen
CREATE OR REPLACE FUNCTION wendat__b_d_reklam() RETURNS TRIGGER AS $$
  BEGIN
    --Wenn QAB bezogener Wareneingang ist, schreiben wir die Wareneingangsnummer in den QAB zurück
    UPDATE qab SET q_w_wen = NULL WHERE q_nr = old.w_q_nr AND NOT q_def AND old.w_lds_id IS NULL;
    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__b_d_reklam
    AFTER DELETE
    ON wendat
    FOR EACH ROW
    WHEN (old.w_q_nr IS NOT NULL)
    EXECUTE PROCEDURE wendat__b_d_reklam();
--

-- WEB anlegen, gegebenenfalls mit QAB verknüpfen
CREATE OR REPLACE FUNCTION wendat__a_iu_check_web() RETURNS TRIGGER AS $$
  DECLARE webnr INTEGER;
          needweb BOOLEAN;
  BEGIN
    IF current_user = 'syncro' THEN RETURN new; END IF;

    -- WEB wird angelegt, wenn es noch keinen WEB für den Wareneingang gibt und im Artikel WEB Prüfungen hinterlegt sind oder der Lagerort einen WEB erfordert
    -- Im Wendat-Trigger, damit Update-Fall bei Mehrfachlagerzugängen berücksichtigt wird. Da gibt es nur UPDATE auf lag und kein Insert.

    -- Prüfung auf Lagerort erfordert WEB
    needWEB := lgort_doweb(new.w_lgort);

    -- Es gibt WEB-Prüfmerkmale im Artikelstamm ... (bei LOLL deaktiviert, siehe www.redmine.prodat-sql.de/issues/7110 - Abklärung QS u. Lagerprozesse / WEBs)
    IF TSystem.Settings__Get('KUNDE') NOT IN ('LOLL', 'LOLL-MECHATRONIK') THEN
        needWEB :=
          needWEB
          OR (
            EXISTS(SELECT true FROM artPruefung JOIN art ON ak_nr = new.w_aknr
                    WHERE (apr_aknr = ak_nr OR (apr_aknr IS null AND apr_ac = ak_ac) )  -- Artikel / AC verlangt WEB (auch umgeschriebener Artikel bei UPDATE-Fall)
                      AND apr_web
                  )
            AND NOT EXISTS(SELECT true FROM ldsdok WHERE ld_id = new.w_lds_id AND ld_code = 'I')  -- WE kommt nich aus interner Buchung
            AND new.w_l_nr IS NULL                                                                -- WE ist keine Materialrückführung
          )
        ;
    END IF;

    -- Prüfung, noch kein WEB angelegt.
    needWEB := needWEB AND NOT EXISTS(SELECT true FROM wareneingangskontrolle WHERE wek_w_wen = new.w_wen);

    IF needWEB THEN
        SELECT getnumcirclenr('web')::INTEGER INTO webnr;
        INSERT INTO wareneingangskontrolle (wek_nr, wek_w_wen, wek_ak_nr, wek_l_krz, wek_dat, wek_q_nr)
            VALUES (webnr, new.w_wen, new.w_aknr, new.w_l_krz, current_date,
                   (SELECT q_nr FROM qab WHERE q_w_wen = new.w_wen AND NOT q_def ORDER BY q_nr DESC LIMIT 1));
        -- Durch das automatische QAB-Erstellen kann es prinipiell mehrere geben, aber immer nur einen offenen QAB.
        PERFORM CopyArtTests('wareneingangskontrolle', webnr, NULL); -- Artikelprüfungen für den Artikel noch umkopieren

        PERFORM PRODAT_HINT(lang_text(16533) || E'\n\n' || webnr);
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_iu_check_web
    AFTER INSERT OR UPDATE
    OF w_l_krz, w_aknr, w_lds_id, w_lgort, w_lagort, w_lagbereich, w_zugang, w_zugang_uf1, w_zug_mec, w_l_nr
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_iu_check_web();
--

--Bei Erstellen oder Aktualisierung einer Auswärtsrückbuchung, aktualisieren wir den Rueckmelde-Eintrag
CREATE OR REPLACE FUNCTION wendat__a_iu_update_rm() RETURNS TRIGGER AS $$
  DECLARE a2id INTEGER;
  BEGIN
    -- ID des AW-Arbeitsgang raussuchen
    SELECT ld_a2_id INTO a2id FROM ldsdok WHERE ld_id = new.w_lds_id AND ld_a2_id IS NOT NULL;
    IF a2id IS NULL THEN
        RETURN new;
    END IF;

    PERFORM tplanterm.update_rm_ausw(a2id, new.w_wen);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_iu_update_rm
    AFTER INSERT OR UPDATE OF w_wen, w_lds_id, w_zug_dat, w_zugang, w_zugang_uf1
    ON wendat
    FOR EACH ROW
    WHEN ((new.w_lds_id IS NOT NULL) AND (new.w_l_krz<>'#'))
    EXECUTE PROCEDURE wendat__a_iu_update_rm();
--

-- DMS - Verschlagwortung
CREATE OR REPLACE FUNCTION wendat__a_iu_keywords() RETURNS TRIGGER AS $$
  DECLARE rec RECORD;
  BEGIN
    -- Nur wenn unbedingt notwendig, das läuft ziemlich oft
    IF tg_op = 'UPDATE' THEN
        IF  (old.w_wen IS NOT DISTINCT FROM new.w_wen)
            AND (old.w_l_krz   IS NOT DISTINCT FROM new.w_l_krz)
            AND (old.w_aknr    IS NOT DISTINCT FROM new.w_aknr)
            AND (old.w_lds_id  IS NOT DISTINCT FROM new.w_lds_id)
            AND (old.w_lgort   IS NOT DISTINCT FROM new.w_lgort)
            AND (old.w_lgchnr  IS NOT DISTINCT FROM new.w_lgchnr)
            AND (old.w_lfsnr   IS NOT DISTINCT FROM new.w_lfsnr)
            AND (old.w_q_nr    IS NOT DISTINCT FROM new.w_q_nr)
            AND (old.w_ks      IS NOT DISTINCT FROM new.w_ks)
            AND (old.w_lbw     IS NOT DISTINCT FROM new.w_lbw)
        THEN
            RETURN new;
        END IF;
    END IF;

    --Lieferscheindokument: diese Schlüsselworte eintragen falls bereits da (Dokument war mit anderer Pos bereits gescannt, diese kommt jetzt zu den bestehenden Dokument hinzu)
    IF (new.w_lfsnr IS NOT NULL) And ( LENGTH(COALESCE(new.w_lfsnr,'')) > 1 ) THEN --Nur bei Länge >1, oft ist "0" Kennzeichen das kein Lieferschein kommt
        --dokumentaktualisierung anstossen, damit dieses die schlüsselworte neu holt (und damit neue datensätze als schlagworte aufnimmt)
        FOR rec IN SELECT pd_id FROM recnokeyword JOIN picndoku ON picndoku.dbrid=r_dbrid
                WHERE  r_kategorie  IN ('lifsch', 'lifscha')
                AND r_descr=new.w_lfsnr
                AND EXISTS(SELECT true FROM recnokeyword r1
                           WHERE r1.r_dbrid=picndoku.dbrid
                             AND r1.r_kategorie='adk'
                             AND r1.r_descr=new.w_l_krz)
        LOOP
            UPDATE picndoku SET pd_id = pd_id WHERE pd_id = rec.pd_id;
        END LOOP;
    END IF;

    IF (tg_op='UPDATE') THEN --Das ist neu und ich bin nich sicher, dass das so geht.
        IF (old.w_lgchnr <> new.w_lgchnr) THEN
            UPDATE picndoku SET pd_id = pd_id WHERE pd_tablename = 'wendat' AND pd_dbrid=new.dbrid;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  -- Bei Ergänzung von UPDATE OF Felder die Triggerfunktion anpassen wg. Prüfung auf Änderung!
  CREATE TRIGGER wendat__a_iu_keywords
    AFTER INSERT OR UPDATE
    OF w_wen, w_l_krz, w_aknr, w_lds_id, w_lgort, w_lgchnr, w_lfsnr, w_q_nr, w_ks, w_lbw
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_iu_keywords();
--

--
CREATE OR REPLACE FUNCTION wendat__a_50_iu__folgeabk() RETURNS trigger AS $$
  -- #9096 bei Folgearbeit sofort nach LZ auftragsbezogen, direkt LA auf ABK automatisch buchen
  DECLARE r RECORD;
  BEGIN
    --zur externen Bestellung (ldsdok E) den PA suchen (ldsdok I)
    SELECT
      (((best.ld_code = 'E') AND TSystem.ENUM_GetValue(best.ld_stat, 'F-ABK')) AND -- externe Bestellung
       ((pa.ld_code = 'I') AND TSystem.ENUM_GetValue(pa.ld_stat, 'F-ABK') AND (pa.ld_abk IS NOT NULL)) -- Produktiosauftrag (PA)
      ) AS isFABK,
      pa.ld_abk AS paABK,
      pa.ld_aknr AS paAKNR,
      pa.ld_ag_id AS paAGID,
      pa.ld_id AS paLDID,
      pa.ld_stk_uf1 AS paSTKUF1 -- Menge bestellt
    INTO
      r
    FROM
      ldsdok AS best
      LEFT JOIN TWawi.ldsdoki__from__ldsdok__folgeabk(best.ld_id) AS pa ON TRUE
    WHERE
      (best.ld_id = new.w_lds_id);

    IF r.isFABK THEN -- nur wenn es eine FolgeABK ist ------------------------------------------------------------- <<<< -------- theoretisches TRIGGER WHEN

      /*
      -- #9582; Bedarf/Deckung wieder herstellen;
      --> wird jetzt in eigenem Trigger gemacht, wenn ldsdok E --> ld_done, wegen Teillieferungen; siehe ldsdok__a_72_iu__ld_folgeap_op_ix

      IF (TG_OP = 'INSERT') THEN
        -- PA wird wieder bedarfsdeckend
        UPDATE ldsdok
        SET
          ld_nbedarf = false
        WHERE
          (ld_id = r.paLDID) AND
          (ld_aknr = r.paAKNR) AND
          ld_nbedarf;

        -- Material (aber nur die eigentlich Bestellte Artikelnummer) wird wieder bedarfswirksam
        UPDATE auftgmatinfo
        SET
          agmi_beistell = false
        WHERE
          (agmi_ag_id = r.paAGID) AND
          agmi_beistell;
      END IF;
      */

      -- Lagerabgang auf ABK entweder einfügen oder ändern

      IF (TG_OP = 'INSERT') THEN -- LA auf ABK anlegen
        -- todo: ggf. in FUNKTION überführen!!
        INSERT INTO lifsch
          (l_krz, l_krzl, l_krzf, l_ag_id , l_ab_ix, l_lgort    , l_lgchnr    , l_aknr  , l_abg_mec    , l_abgg      , l_inliefdok)
        VALUES
          ('#'  , '#'   , '#'   , r.paAGID, r.paABK, new.w_lgort, new.w_lgchnr, r.paAKNR, new.w_zug_mec, new.w_zugang, false)
        RETURNING l_nr INTO new.w_l_nr;
      ELSE  -- (TG_OP = 'UPDATE') -- LA auf ABK ändern (z.B. LZ-Menge wurde im Modul "LZ-Bearbeiten" geändert)
        UPDATE lifsch
        SET
          l_lgort = new.w_lgort,
          l_lgchnr = new.w_lgchnr,
          l_abg_mec = new.w_zug_mec,
          l_abgg = new.w_zugang,
          l_abgg_uf1 = new.w_zugang_uf1
        WHERE
          (l_ab_ix = r.paABK) AND -- zur Sicherheit
          (l_aknr = r.paAKNR) AND -- zur Sicherheit
          (l_w_wen = new.w_wen) AND -- exakte Beziehung
          ((l_abgg <> new.w_zugang) OR -- bei geänderter Menge oder
           (l_abgg_uf1 <> new.w_zugang_uf1) OR
           (l_lgort <> new.w_lgort) OR -- bei geändertem Lagerort oder
           (l_lgchnr <> new.w_lgchnr) OR -- bei geänderter Chargennummer oder
           (l_abg_mec <> new.w_zug_mec)); -- bei geänderter Mengeneinheit

        -- Den Lagerist darauf hinweisen, das hier eine Folge-ABK zu beachten ist
        PERFORM PRODAT_MESSAGE(lang_text(30150), 'Information'); --'ACHTUNG! Folgearbeit (Folge-ABK) beachten!'
      END IF;

      -- #10676 Materialposition schließen, wenn ausreichend Menge geliefert
      UPDATE auftg
      SET
        ag_done = TRUE
      WHERE TRUE
        AND (ag_astat = 'I')
        AND (not ag_done)
        AND (ag_id = r.paAGID)
        AND ((SELECT SUM(l_abgg_uf1) /*SUM(l_abgg)*/ FROM lifsch WHERE l_ag_id = r.paAGID) >= r.paSTKUF1);

    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_50_iu__folgeabk
    AFTER INSERT OR UPDATE
    OF w_zugang, w_zugang_uf1, w_lgort, w_lagbereich, w_lgchnr, w_zug_mec
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE public.wendat__a_50_iu__folgeabk();
--

--
CREATE OR REPLACE FUNCTION wendat__b_50_d__folgeabk() RETURNS trigger AS $$
-- #9096 bei Folgearbeit sofort nach LZ direkt LA auf ABK automatisch buchen
  DECLARE r RECORD;
  BEGIN
    --zur externen Bestellung (ldsdok E) den PA suchen (ldsdok I)
    SELECT
      (((best.ld_code = 'E') AND TSystem.ENUM_GetValue(best.ld_stat, 'F-ABK')) AND -- externe Bestellung
       ((pa.ld_code = 'I') AND TSystem.ENUM_GetValue(pa.ld_stat, 'F-ABK') AND (pa.ld_abk IS NOT NULL)) -- Produktiosauftrag (PA)
      ) AS isFABK,
      pa.ld_abk AS paABK,
      pa.ld_aknr AS paAKNR
      --,pa.ld_ag_id AS paAGID
    INTO
      r
    FROM
      ldsdok AS best
      LEFT JOIN TWawi.ldsdoki__from__ldsdok__folgeabk(best.ld_id) AS pa ON TRUE
    WHERE
      (best.ld_id = old.w_lds_id);

    IF r.isFABK THEN -- nur wenn es eine FolgeABK ist ------------------------------------------------------------- <<<< -------- theoretisches TRIGGER WHEN

      -- wenn der LZ gelöscht wird, dann auch den LA auf die ABK entfernen

      DELETE FROM
        lifsch
      WHERE
        (l_ab_ix = r.paABK) AND -- zur Sicherheit
        (l_aknr = r.paAKNR) AND -- zur Sicherheit
        (l_w_wen = old.w_wen); -- exakte Beziehung

      IF NOT found THEN
        RETURN NULL;
      END IF;
    END IF;

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__b_50_d__folgeabk
    BEFORE DELETE
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE public.wendat__b_50_d__folgeabk();
--

-- Liefermenge Bestellung, Lag updaten wenn bestandswirksam
CREATE OR REPLACE FUNCTION wendat__a_d() RETURNS TRIGGER AS $$
  DECLARE _wendatbestandswirksam boolean;
  BEGIN
    -- für NICHT-UE Buchungen Menge geliefert zurücksetzen
    IF old.w_a2_id IS NULL THEN
       PERFORM disablemodified();
       UPDATE ldsdok SET ld_stkl = ld_stkl - old.w_zugang_uf1 WHERE ld_id = old.w_lds_id;
       PERFORM enablemodified();
    ELSE
       -- w_a2_id > es ist ein Bezug zur abk vorhanden. Lagerbuchung UE
       -- Materialposition für UE entfernen, da Lagzugang auch entfernt wird
       DELETE FROM auftg
             WHERE ag_id = (SELECT ag_id
                              FROM auftg, auftgmatinfo
                             WHERE agmi_ag_id = ag_id
                               AND ag_parentabk = old.w_ab_ix
                               AND ag_a2_id = old.w_a2_id
                               AND agmi_lg_chnr = old.w_lgchnr
                            )
               AND Enum_GetValue(ag_stat, 'UE')
       ;

       IF found THEN
          PERFORM prodat_hint('delete wendat also delete auftgi UE xtt21741');
       ELSE
          PERFORM prodat_error('delete wendat UE BUT NO auftgi found xtt21742');
       END IF;
    END IF;


    /******************************** HIER IST SCHLUSS, WENN AUSWAERTSRUECKBUCHUNG ODER ARTIKEL NICHT LAGERGEFÜHRT ODER ARBEITSPAKET ********************/
    IF (NOT TArtikel.art__ac_i__is__50(old.w_aknr)) THEN
        RETURN old;
    END IF;
    --
    _wendatbestandswirksam := tlager.wendatchange_checkbestwirksam(old);

    --alten Lagerort zurückrechnen
    IF _wendatbestandswirksam THEN

        -- Lager UE: kein Lagerlog durch lag-Änderung!
        IF NOT old.w_a2_id IS NULL THEN
           PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'F'); -- Lagerlogging abschalten, wird hier in dem Trigger geführt
        END IF;

        UPDATE lag SET lg_anztot = lg_anztot - old.w_zugang_uf1
         WHERE lg_id = TLager.lag__lg_id__from__wendat(old);

        IF old.w_a2_id IS NULL THEN
           PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'T');
        END IF;

        IF NOT found AND NOT TSystem.Settings__GetBool('no_neg_lag') THEN
           --alter Lagerort nicht mehr vorhanden, dann anlegen!
           --nur wirksam wenn negative Bestände erlaubt, weil sonst sowieso unter 0
           PERFORM TLager.lag__from__wendat__create(old, true);
        END IF;
    END IF;
    --
    PERFORM TLager.lagerlog__create('=+'||IFTHEN(_wendatbestandswirksam, '', ' !'), NULL, old); --- #14796 wendat__a_d()

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_d
    AFTER DELETE
    ON wendat
    FOR EACH ROW
    EXECUTE PROCEDURE wendat__a_d();
--


CREATE OR REPLACE FUNCTION wendat__a_90_iu__w_a2_id() RETURNS TRIGGER AS $$

  -- #16116 Füllt die Materialliste mit einem Halbfertigartikel, wenn dieser zugebucht wird

  DECLARE
      _ab_ix integer;
  BEGIN

      IF (      tg_op = 'INSERT'
           OR ( tg_op = 'UPDATE' AND old.w_a2_id IS null )
           )
      THEN
         _ab_ix := a2_ab_ix FROM ab2 WHERE a2_id = new.w_a2_id;
         PERFORM teinkauf.auftg__folgebedarf__insert( new.w_lds_id, _ab_ix, new.w_zugang, 'UE', new.w_wen );
      END IF;

      RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER wendat__a_90_iu__w_a2_id
    AFTER INSERT OR UPDATE OF w_a2_id
    ON wendat
    FOR EACH ROW
    WHEN ( new.w_a2_id IS NOT null )
    EXECUTE PROCEDURE wendat__a_90_iu__w_a2_id();
--

-- ENDE WARENEINGANGSSCHEIN ----------------------------------------------------------------------------



--Definitionstabelle für Zwischenlagerorte
CREATE TABLE lagerortUE
  (
   lue_agnr             VARCHAR(20) NOT NULL,
   lue_lgort            VARCHAR(50) NOT NULL
  );

-- Warenausgang
CREATE TABLE lifsch (
  l_nr                 SERIAL PRIMARY KEY,
  l_krz                VARCHAR(21) NOT NULL REFERENCES adk ON UPDATE CASCADE,           --
  l_krzl               VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --
  l_krzf               VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, --
  l_aknr               VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE,           -- Gelieferter Artikel
  l_ag_id              INTEGER CONSTRAINT xtt5115 REFERENCES auftg,                     -- Auftrags-ID
  l_ab_ix              INTEGER,                                                         -- REFERENCES abk
  l_an_nr              VARCHAR(50) REFERENCES anl,                                          -- Anlagen-/Projektnummer
  l_ldat               TIMESTAMP(0) WITHOUT TIME ZONE NOT NULL DEFAULT currenttime(),
  l_kdat               DATE,                                                            -- eigentlich bestätigter Liefertermin!
  l_akdat              DATE,                                                            -- Datum Warenausgang
  l_bda                VARCHAR(50),                                                     -- Bestellnummer Kunde
  l_lgort              VARCHAR(50) NOT NULL DEFAULT '',
  l_lgchnr             VARCHAR(50) NOT NULL DEFAULT '',
  l_abg_mec            INTEGER NOT NULL CONSTRAINT xtt4064 REFERENCES artmgc,           -- abgehende Mengeneinheit
  l_abgg               NUMERIC(14,6) NOT NULL CONSTRAINT xtt5111__l_abgg CHECK (l_abgg>=0),   -- abgehende Menge
  l_abgg_uf1           NUMERIC(20,8),
  -- l_stkl_uf1          NUMERIC(12,4) NOT NULL DEFAULT 0,                              -- (LG,12/2013 - Obsolet, keine Teilmengenübernahme mehr) Menge die in Lieferdokument übernommen wurde (In Grundmengeneinheit)
  l_inliefdok          BOOLEAN NOT NULL DEFAULT FALSE,                                  -- Der Lagerabgang wurde auf einem Lieferdokument erfasst (oder soll nicht und gilt als übernommen)
  l_vkp_uf1            NUMERIC(20,8),
  l_hest               NUMERIC(12,4),
  l_vkpbas             NUMERIC(12,4),
  l_vkp                NUMERIC(12,4),
  l_arab               NUMERIC(5,2),
  l_azutx              TEXT,
  l_azutx_rtf          TEXT,
  l_gew                NUMERIC(14,6),
  l_br_tafel           INTEGER,
  l_versart            VARCHAR(30),
  -- l_dokunr           INTEGER,
  -- l_dokupos          INTEGER,
  -- l_def              BOOL DEFAULT FALSE,
  -- l_fakt             NUMERIC NOT NULL DEFAULT 0,                                     -- Wird aus belegpos_a_iud gesetzt, wenn Liefeschein auf Dokument ist, aus dem in eine Rechnung uebernommen wurde.
  l_dofakt             BOOL NOT NULL DEFAULT TRUE,                                      -- verrechenbar (zB nicht werthaltiger Lagerort)
  l_dim1               NUMERIC(12,4) NOT NULL DEFAULT 0,
  l_dim2               NUMERIC(12,4) NOT NULL DEFAULT 0,
  l_dim3               NUMERIC(12,4) NOT NULL DEFAULT 0,
  l_qab                INTEGER,                                                         -- Lieferschein wurde für Retourlieferung durch diesen QAB erstellt <= wird aus TFormQSRetour gesetzt beim erstellen Retourelieferschein
  l_w_wen              INTEGER REFERENCES wendat ON UPDATE CASCADE,                     -- Retourelieferung=>Retourlieferschein aus QAB bezog sich auf diese Auswärtsrücklieferung oder normalen Lagerzugang  <= wird aus TFormQSRetour gesetzt beim erstellen Retourelieferschein
                                                                                        -- Normale Lieferung: wenn leer, übernommen aus Lagerort, von dem der Lagerabgang ausgeführt wurde
                                                                                        -- ACHTUNG: BEACHTE w_l_nr: Rücklieferung auf eine Auslieferung (Verringerung) http://redmine.prodat-sql.de/issues/6327
  l_ld_id              INTEGER REFERENCES ldsdok ON UPDATE CASCADE,                     -- Bestellung, welche die Ware geliefert hat, wird automatisch aus Lagerort übernommen, bzw. aus Wendat wenn l_w_wen
  l_belp_id            INTEGER, -- REFERENCES belegpos ON UPDATE CASCADE ON DELETE SET NULL -- siehe X TableContraints
                                                                                        -- Der Lagerabgang ist in diese Lieferscheinposition eingegangen. Wenn das Feld gesetzt ist, gehen wir davon aus,
                                                                                        -- dass die komplette Lagerabgangsmenge in den Lieferschein übernommen wurde. Keine Teilmengen bei Lieferscheinpos. aus mehreren Lagerabgängen.
  l_bz_id              INTEGER, -- REFERENCES belzeil_auftg_lif(bz_id) ON UPDATE CASCADE -- siehe X TableContraints
                                                                                        -- Der LA ist einer Rechnung zugeordnet, vgl. vereinfachten Konsignationsprozess, siehe #8093

  -- l_qab_ldid(_obsolet)      INTEGER,                                                    -- Umgesetzt auf w_wen, Retour bezog sich auf diese Bestellung/Wareneingang
  -- l_qab_awrid(_obsolet)     INTEGER,                                                    -- Alt, (wahrscheinlich) unbenutzt Retourelieferung bezog sich auf diese Auswärtsbearbeitung, entfällt nach Auswärtsumsetzung
  l_lgort_ue           VARCHAR(50),                                                     -- Lagerort UE, gibt an wo das Zeug nach Lagerabgang für Fertigungsauftrag in der Werkhalle liegt, kein Lagerort aus "lag"
                                                                                        -- beinhaltet auch den Lagerort ('KONSI%'||Adresskürzel) beim Konsignationslagerprozess
  l_ks_abt             VARCHAR(9) REFERENCES ksv,                                       -- Lagerabgang Zuordnung zu Kostenstelle (zB Kühlmittelverbrauch auf Maschine)
  l_konsDat            DATE,                                                            -- Konservierungsdatum des Lagerortes zum Zeitpunkt des LA (Sicherung)
  l_konsEnde           DATE,                                                            -- Ablauf der Konservierung des Lagortes zum Zeitpunkt des LA (Sicherung)
  l_la_id              INTEGER REFERENCES ldsauftg ON UPDATE CASCADE ON DELETE SET NULL
);

SELECT setval('lifsch_l_nr_seq', 10000);

-- Indizes
    --CREATE INDEX lifsch_seldoku ON lifsch(l_krzl, l_dokunr);
    CREATE INDEX lifsch_l_krzl_like ON lifsch(l_krzl varchar_pattern_ops);
    CREATE INDEX lifsch_l_krzf ON lifsch(l_krzf);
    CREATE INDEX lifsch_l_krzf_like ON lifsch(l_krzf varchar_pattern_ops);
    CREATE INDEX lifsch_l_krz ON lifsch(l_krz);
    CREATE INDEX lifsch_l_krz_like ON lifsch(l_krz varchar_pattern_ops);
    --CREATE INDEX lifsch_dokunr ON lifsch(l_dokunr) WHERE l_dokunr IS NOT NULL;
    CREATE INDEX lifsch_ldat ON lifsch(CAST(l_ldat AS DATE));
    CREATE INDEX lifsch_ag_id ON lifsch(l_ag_id);
    CREATE INDEX lifsch_ag_an_nr ON lifsch(l_an_nr);
    CREATE INDEX lifsch_ag_an_nr_like ON lifsch(l_an_nr varchar_pattern_ops);
    CREATE INDEX lifsch_l_bda ON lifsch(l_bda);
    CREATE INDEX lifsch_l_bda_like ON lifsch(l_bda varchar_pattern_ops);
    CREATE INDEX lifsch_l_br_tafel ON lifsch(l_br_tafel) WHERE l_br_tafel IS NOT NULL;
    CREATE INDEX lifsch_l_aknr ON lifsch(l_aknr);
    CREATE INDEX lifsch_l_aknr_like ON lifsch(l_aknr varchar_pattern_ops);
    CREATE INDEX lifsch_date_to_yearmonth_ldat ON lifsch(date_to_yearmonth(l_ldat));
    CREATE INDEX lifsch_l_belp_id ON lifsch (l_belp_id) WHERE l_belp_id IS NOT NULL;
    CREATE INDEX lifsch_l_qab ON lifsch(l_qab) WHERE l_qab IS NOT NULL;
    CREATE INDEX lifsch_l_ab_ix ON lifsch(l_ab_ix) WHERE l_ab_ix IS NOT NULL;
    CREATE INDEX lifsch_l_la_id ON lifsch(l_la_id) WHERE l_la_id IS NOT NULL;
--

--CREATE SEQUENCE lifsch_l_dokunr_seq;

-- Mengeneinheit in Umrechnungsfaktor
CREATE OR REPLACE FUNCTION lifsch__b_i() RETURNS TRIGGER AS $$
  DECLARE
    _opix integer;
    _agstat varchar;
    _beistellmaterial boolean;
    _ag_aknr varchar;
    _ag_id integer;
  BEGIN
    SELECT ak_hest, ak_vkpbas
      INTO new.l_hest, new.l_vkpbas
      FROM art
    WHERE ak_nr = new.l_aknr;

    IF new.l_ag_id IS NOT null THEN

        SELECT -- Wenn Auftragsbezug vorhanden und nur l_ag_id, dann defaults aus auftg
               coalesce(new.l_aknr,    ag_aknr),
               -- coalesce(new.l_abgg,    ag_stk), -- NICHT. Die Menge MUSS eingegeben werden, zu hohe Fehlerquelle wenn in Oberfläche einfach speichern gedrückt wird.
               coalesce(new.l_abg_mec, ag_mcv),
               coalesce(new.l_krz,     ag_lkn),
               coalesce(new.l_krzl,    ag_krzl),
               coalesce(new.l_krzf,    ag_krzf),
               ag_vkp,
               ag_vkp_uf1,
               ag_arab,
               ag_ldatum,
               ag_bda,
               ag_stat,
               coalesce(agmi_beistell, false),
               ag_aknr
          INTO new.l_aknr,
               -- new.l_abgg,
               new.l_abg_mec,
               new.l_krz,
               new.l_krzl,
               new.l_krzf,
               new.l_vkp,
               new.l_vkp_uf1,
               new.l_arab,
               new.l_kdat,
               new.l_bda,
               _agstat,
               _beistellmaterial,
               _ag_aknr
        FROM auftg
          LEFT JOIN auftgmatinfo ON agmi_ag_id = ag_id
        WHERE ag_id = new.l_ag_id;
      ELSE
        IF new.l_abg_mec IS null THEN
           new.l_abg_mec := tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( new.l_aknr );
        END IF;
    END IF;

    new.l_krzl := coalesce( new.l_krzl, new.l_krz );
    new.l_krzf := coalesce( new.l_krzf, new.l_krz );

    -- #18078 Auftragsposition hinzufügen, wenn Alternativartikel
    -- Wird eine Alternativposition ausgelagert, dann wird eine neue, abgeschlossene Auftragsposition für den Alternativartikel
    -- angelegt und mit dem Status "AM" (AlternativArtikel/Material) versehen. Die Auftragsmenge der bisherigen Position wird
    -- entsprechend reduziert. Diese Position ist nun die Auftragspositon des Lagerabgangs.
    IF new.l_aknr <> coalesce(_ag_aknr, new.l_aknr) THEN -- wenn die Artikelnummer ungleich Auftrag (ohne Auftrag ist _ag_aknr NULL)

       -- Vom Auftrag abweichende Artikelnummer muss ein Alternativartikel sein!
       IF new.l_aknr NOT IN ( SELECT tartikel.artoption_arts__aknr__get( _ag_aknr, true )) THEN
          RAISE EXCEPTION 'lifsch__b_i__art_isnot_artoption_arts';
       END IF;

       INSERT INTO auftg
                   ( ag_astat, ag_nr, ag_hpos, ag_aknr, ag_stk, ag_txt, ag_parentabk, ag_mainabk, ag_stat,
                     ag_lkn, ag_krzl, ag_krzf, ag_ldatum, ag_bda, ag_done )
            SELECT
                   ag_astat, ag_nr, ag_pos, ak_nr, new.l_abgg, ag_txt, ag_parentabk, ag_mainabk, 'AM',
                   ag_lkn, ag_krzl, ag_krzf, ag_ldatum, ag_bda, true
              FROM auftg
              JOIN art ON ak_nr = new.l_aknr
             WHERE ag_id = new.l_ag_id
         RETURNING ag_id INTO _ag_id;

       -- Ausgangs-Auftragsposition (für diese wurde jetzt die alternative eingefügt) um die Menge reduzieren, welche wir jetzt altantivmaterial abgebucht haben
       UPDATE auftg
          SET ag_stk = ag_stk - me__menge_uf1__in__menge(ag_mcv, new.l_abgg_uf1)
        WHERE ag_id = new.l_ag_id;

       -- der Lagerabgang hat Bezug zur neuen Alternativmaterial-Auftragsposition
       new.l_ag_id := _ag_id;

    END IF;

    -- Prüfplan beachten
    IF NOT current_user = 'syncro' THEN
        SELECT op_ix INTO _opix FROM opl WHERE op_n = new.l_aknr AND op_standard; -- Standardvariante Fertigung suchen
        IF _opix IS NOT NULL THEN -- Datum letzte Lieferung im Prüfplan aktualisieren
            PERFORM disablemodified();
            UPDATE op8 SET o8_expdatedlv= current_date + o8_validmonthdlv * '1 month'::INTERVAL - '1 day'::INTERVAL
            WHERE o8_ix = _opix AND o8_validmonthdlv IS NOT NULL AND o8_active;
            PERFORM enablemodified();
        END IF;
    END IF;
    --
    IF new.l_ag_id IS NOT NULL AND new.l_ab_ix IS NULL THEN
        new.l_ab_ix:= ag_parentabk FROM auftg WHERE ag_id = new.l_ag_id;
    END IF;

    IF new.l_ag_id IS NOT NULL AND NOT TSystem.Enum_GetValue( _agstat, 'F-ABK' ) THEN -- #8566, #9582, #10102 Automatischer Lagerabgang ABK-bezogen für FolgeABK zulassen, obwohl er Beistellung (Krücke für: nicht Bedarfswirksam) ist
        IF _beistellmaterial AND NOT COALESCE((lagerorte__get_setup(new.l_lgort))._beistell, false) THEN -- sonst bei Beistellung einen Beistelllagerort verlangen
            RAISE EXCEPTION '%', lang_text(11172);
        END IF;
    END IF;
    -- wenn von einem nicht werhaltigen Lagerort abgebucht wird, ist standard keine Ausgangsrechnung (Retourelieferung)
    IF new.l_qab IS NOT NULL -- Retourelieferung auf QAB
        OR NOT COALESCE((lagerorte__get_setup(new.l_lgort))._werthg, true)
    THEN
        new.l_dofakt:= false;
    END IF;
    --

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__b_i
    BEFORE INSERT
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__b_i();
--

--
CREATE OR REPLACE FUNCTION lifsch__b_iu() RETURNS TRIGGER AS $$
  DECLARE isSperrLager BOOLEAN;
          lag_rec RECORD;
          lag_info_need_upd BOOLEAN;
  BEGIN

    -- Sperrlagerort ?
    SELECT (TLager.lag__from__lifsch(new)).lg_sperr INTO isSperrLager;

    -- ... und kein QS Lagerabgang? (also Retoure an Lieferant oder Abbuchung auf QS-Auftrag)
    IF         isSperrLager
       AND     new.l_qab IS NULL
       AND NOT EXISTS( SELECT true
                        FROM auftg
                       WHERE ag_id = new.l_ag_id
                         AND (   ag_q_nr IS NOT NULL
                                 -- Lagerabgang von UE-Buchungen ist auf Sperrlager zulässig
                              OR TSystem.ENUM_GetValue(ag_stat, 'UE')
                              )
                      )
    THEN
        RAISE EXCEPTION '% % % %', lang_text(4800)||': ',new.l_lgort,' '||lang_text(2483)||': ',new.l_lgchnr;
    END IF;

    -- Der Datensatz wurde mit einer für den Artikel ungültigen Mengeneinheit gespeichert. Tabelle: %, ID: %, Feld: %, ART-Nr.: %, Artmgc-ID: %
    IF new.l_aknr IS NOT NULL AND NOT TArtikel.me__art__artmgc__m_ids__valid(new.l_aknr, new.l_abg_mec) THEN
       RAISE EXCEPTION '%', Format(lang_text(13795), 'lifsch', new.l_nr::VARCHAR, 'l_abg_mec', new.l_aknr, new.l_abg_mec::VARCHAR);
    END IF;

    --
    new.l_abgg_uf1 := tartikel.me__menge__in__menge_uf1(new.l_abg_mec, new.l_abgg);

    IF TG_OP = 'UPDATE' THEN
        -- Lieferscheinpos. löschen => l_belp_id wird NULL gesetzt => Lagerabgang wieder für Übernahmen öffnen
        IF (old.l_belp_id IS NOT NULL AND new.l_belp_id IS NULL) AND (old.l_inliefdok) THEN
            new.l_inliefdok:= false;
        END IF;
        -- Buchung von anderem Lagerort, umschreiben der Informationen aus Lagerort für LA nötig (s.u.)
        IF    (new.l_aknr IS DISTINCT FROM old.l_aknr)
           OR (new.l_lgort IS DISTINCT FROM old.l_lgort)
           OR (new.l_lgchnr IS DISTINCT FROM old.l_lgchnr)
           OR (new.l_dim1 IS DISTINCT FROM old.l_dim1)
           OR (new.l_dim2 IS DISTINCT FROM old.l_dim2)
           OR (new.l_dim3 IS DISTINCT FROM old.l_dim3)
        THEN
            lag_info_need_upd:= true;
        END IF;
    END IF;
    --
    IF current_user='syncro' THEN
        RETURN new;
    END IF;

    --Projektnummer aus Auftrag
    IF new.l_ag_id IS NOT NULL THEN
        new.l_an_nr:=COALESCE(ag_an_nr, new.l_an_nr) FROM auftg WHERE ag_id=new.l_ag_id;
    ELSE
        new.l_an_nr:=NULL;
    END IF;
    --

    -- Informationen aus Lagerort für LA hinterlegen (Wareneingangsbezug, Bestellbezug, Konservierungsdatum, Ablauf der Konservierung).
    IF TG_OP = 'INSERT' OR COALESCE(lag_info_need_upd, false) THEN
        SELECT lg_id, lg_w_wen, lg_ld_id, lg_konsDat, lg_konsEnde
          INTO lag_rec
          FROM TLager.lag__from__lifsch(new);

        IF lag_rec.lg_id IS NOT NULL THEN -- Lagerort existiert, dann umschreiben
            new.l_w_wen:=    lag_rec.lg_w_wen;
            new.l_ld_id:=    lag_rec.lg_ld_id;
            new.l_konsDat:=  lag_rec.lg_konsDat;
            new.l_konsEnde:= lag_rec.lg_konsEnde;
        END IF;
    END IF;
    --

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__b_iu
    BEFORE UPDATE OR INSERT
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__b_iu();
--

--Verfügbarkeit in Art setzen
CREATE OR REPLACE FUNCTION lifsch__a_i() RETURNS TRIGGER AS $$
  DECLARE _hnnr VARCHAR;
  BEGIN
    IF (current_user='syncro') THEN
        RETURN new;
    END IF;

    IF new.l_ag_id IS NOT NULL AND new.l_bz_id IS NULL THEN -- LA ist aus Auftrag und keiner Rechnung (auf Konsi-LFS) zugeordnet.
        PERFORM disablemodified();
        UPDATE auftg
           SET ag_stkl = ag_stkl + new.l_abgg_uf1,
               -- Kennzeichen für Lagerabgangsbuchung ausführen. Wir wissen evtl nicht in welchem Statusfeld das steht
               -- Zurücksetzen wenn Abbuchung geschehen (somit kann das Kennzeichen für eine weitere Teillieferung wieder gesetzt werden.)
               ag_bstat  = IFTHEN(TSystem.Settings__GetBool('lifsch__ag_bstat__w__clear', true), TSystem.Enum_DelValue( ag_bstat,  'W' ), ag_bstat),
               ag_bstat1 = IFTHEN(TSystem.Settings__GetBool('lifsch__ag_bstat__w__clear', true), TSystem.Enum_DelValue( ag_bstat1, 'W' ), ag_bstat1),
               ag_bstat2 = IFTHEN(TSystem.Settings__GetBool('lifsch__ag_bstat__w__clear', true), TSystem.Enum_DelValue( ag_bstat2, 'W' ), ag_bstat2)
         WHERE ag_id = new.l_ag_id; -- TODO TSystem_Wawi.beleg_p__verkauf__mengelf__refresh
        PERFORM enablemodified();
    END IF;

    /******************************** HIER IST SCHLUSS, WENN ARTIKEL NICHT LAGERGEFÜHRT ODER ARBEITSPAKET ********************/
    IF (NOT TArtikel.art__ac_i__is__50(new.l_aknr)) THEN
        RETURN new;
    END IF;

    -- ACHTUNG, KONSIGNATIONSPROZESS bucht nicht aus dem Lager aus, sondern macht einen Lagerortwechsel
    -- wenn der LA aus externem Konsi-Auftrag und ohne ABK und nicht über die Rechnung kommt.
    IF EXISTS(SELECT true FROM auftg WHERE ag_id = new.l_ag_id AND ag_astat = 'E' AND 'K' IN (ag_bstat, ag_bstat1, ag_bstat2))
        AND new.l_ab_ix IS NULL -- keine ABK
        AND new.l_bz_id IS NULL -- kein Rechnungsbezug (Rechnung auf Konsi-LFS)
    THEN
        RETURN new;
    END IF;

    --Lag nur updaten wenn Artikel lagerverwaltet und Lieferschein kein Retourlieferschein für Auswärts-QAB
    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'F');


    UPDATE lag
       SET lg_anztot = lg_anztot - new.l_abgg_uf1
     WHERE lg_id =  TLager.lag__lg_id__from__lifsch(new);

    IF NOT found AND NOT TSystem.Settings__GetBool('no_neg_lag') THEN
        -- Lieferschein-Datensatz auf nicht vorhandenen Lagerort angelegt. Ergibt eigentlich keinen Sinn dann jetzt den Lagerort anzulegen nochmal mit der Menge?!
        PERFORM TLager.lag__from__lifsch__create(new, true);
    END IF;

    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'T');

    PERFORM TLager.lagerlog__create('-', new); --- #14796

    --dies geht auch an den HÄNEL-LIFT!!!
    IF EXISTS(SELECT true FROM haenel WHERE new.l_lgort ILIKE hl_lg_ort) AND (new.l_qab IS NULL) THEN
        SELECT ak_nrhl INTO _hnnr FROM art WHERE ak_nr=new.l_aknr;
        INSERT INTO haenel_aktion (ha_aktion, ha_aknr, ha_nr_pos, ha_stk, ha_lgort) VALUES ('-', _hnnr, new.l_nr, new.l_abgg_uf1, new.l_lgort);
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__a_i
    AFTER INSERT
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__a_i();
--

-- Konsignationslagerprozess #7780; #7943; #8093
CREATE OR REPLACE FUNCTION lifsch__a_50_i__konsignationslager() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (new.l_ag_id IS NOT NULL AND new.l_ab_ix IS NULL AND new.l_bz_id IS NULL) -- LA ist aus Auftrag und keiner ABK und keiner Rechnung (auf Konsi-LFS) zugeordnet.

    -- ACHTUNG, KONSIGNATIONSPROZESS bucht nicht aus dem Lager aus, sondern macht einen Lagerortwechsel,
    -- wenn der LA aus externem Konsi-Auftrag und ohne ABK und nicht über die Rechnung kommt, siehe WHEN.
    IF EXISTS(SELECT true FROM auftg WHERE ag_id = new.l_ag_id AND ag_astat = 'E' AND 'K' IN (ag_bstat, ag_bstat1, ag_bstat2)) THEN
        -- Daten in lifsch sollten schon in der Oberfläche gesetzt werden, hier nur noch mal sicherheitshalber zur Korrektur
        new.l_lgort_ue := 'KONSI-' || COALESCE(new.l_krzl, '');   -- ggf. Kommisionierungslager setzen; Kürzel + Lieferadresse
        UPDATE lifsch SET                                         -- UPDATE notwendig, da AFTER-Trigger;
          l_dofakt   = TSystem.Settings__GetBool('EntnahmeVomKonsiLiefsch'), -- "Lieferschein steht nicht zur Fakturierung an" #7780 oder "verschlankt Konsiprozess" #7943
          l_lgort_ue = new.l_lgort_ue                             -- ggf. Kommisionierungslager setzen; Kürzel + Lieferadresse
        WHERE l_nr=new.l_nr AND l_aknr=new.l_aknr;

        PERFORM tlager.lag__ort_chnr__lagerortwechsel(new.l_aknr, --aknr VARCHAR,
                                                      new.l_lgort, --oort VARCHAR,
                                                      new.l_lgchnr, --ochnr VARCHAR,
                                                      new.l_lgort_ue, --nort VARCHAR,
                                                      new.l_lgchnr, -- nchnr VARCHAR,
                                                      new.l_abgg_uf1, --menge NUMERIC,
                                                      COALESCE(new.l_dim1, 0)::INTEGER, --dim1 INTEGER,
                                                      COALESCE(new.l_dim2, 0)::INTEGER, --dim2 INTEGER,
                                                      COALESCE(new.l_dim3, 0)::INTEGER, --dim3 INTEGER,
                                                      true --dellag BOOL, ...
                                                      );
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__a_50_i__konsignationslager
    AFTER INSERT
    ON lifsch
    FOR EACH ROW
    WHEN (new.l_ag_id IS NOT NULL AND new.l_ab_ix IS NULL AND new.l_bz_id IS NULL) -- LA ist aus Auftrag und keiner ABK und keiner Rechnung (auf Konsi-LFS) zugeordnet.
    EXECUTE PROCEDURE lifsch__a_50_i__konsignationslager();
--

--Anpassung von ag_stkl sowie Lagerbeständen, Berücksichtigung RetoureLieferungen, Sperrlager, Bestandswirksame Änderung usw.
CREATE OR REPLACE FUNCTION lifsch__a_u__lagchange() RETURNS TRIGGER AS $$
  DECLARE lifschbestandswirksam BOOL;
  BEGIN
    IF current_user='syncro' THEN--Artikelnummer austauschen
        RETURN new;
    END IF;
    IF ((old.l_abgg_uf1 IS DISTINCT FROM new.l_abgg_uf1) OR (old.l_ag_id IS DISTINCT FROM new.l_ag_id)) -- Mengen- oder Auftragsbezugänderung vorhanden
        AND new.l_bz_id IS NULL                                                                         -- und keiner Rechnung (auf Konsi-LFS) zugeordnet.
    THEN
        PERFORM disablemodified();
        UPDATE auftg SET ag_stkl= ag_stkl - old.l_abgg_uf1 WHERE ag_id = old.l_ag_id;
        UPDATE auftg SET ag_stkl= ag_stkl + new.l_abgg_uf1 WHERE ag_id = new.l_ag_id;
        PERFORM enablemodified();
    END IF;

    /******************************** HIER IST SCHLUSS, WENN ARTIKEL NICHT LAGERGEFÜHRT ODER ARBEITSPAKET ********************/

    -- ACHTUNG, KONSIGNATIONSPROZESS bucht nicht aus dem Lager aus, sondern macht einen Lagerortwechsel
    -- wenn der LA aus externem Konsi-Auftrag und ohne ABK und nicht über die Rechnung kommt.
    IF EXISTS(SELECT true FROM auftg WHERE ag_id = new.l_ag_id AND ag_astat = 'E' AND 'K' IN (ag_bstat, ag_bstat1, ag_bstat2))
        AND new.l_ab_ix IS NULL -- keine ABK
        AND new.l_bz_id IS NULL -- kein Rechnungsbezug (Rechnung auf Konsi-LFS)
    THEN
        PERFORM PRODAT_TEXT(lang_text(30083) || COALESCE(new.l_lgort, ''));   -- Achtung! Konsignations-Lagerort muss manuell korrigiert werden! Lagerort:
        RETURN new;
    END IF;

    IF (new.l_abgg<>old.l_abgg) THEN --Rückbuchung auf einen Lagerabgang > http://redmine.prodat-sql.de/issues/6911 ; http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Lager
        IF EXISTS(SELECT true FROM wendat WHERE w_l_nr=new.l_nr) THEN
            UPDATE auftg SET ag_done=FALSE WHERE ag_id=new.l_ag_id AND ag_done; --wiederöffnen Auftragsposition
            IF FOUND THEN
                PERFORM PRODAT_HINT('Material wieder geöffnet');
            END IF;
            RETURN new;
        END IF;
    END IF;
    --
    IF (NOT TArtikel.art__ac_i__is__50(new.l_aknr)) THEN --Artikel ohne Lagerführung > keine Erhöhung, nur Abbuchung > Verweiß auf Wiki/Ticket?????
        RETURN new;
    END IF;

    /*************************************************************************************************************************/

    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'F');

    lifschbestandswirksam := tlager.lifschchange_checkbestwirksam(new) OR old.l_abgg_uf1 > new.l_abgg_uf1;-- ... oder wir verringern die abgangsmenge, dann wird einfach die differenz zugebucht!

    IF lifschbestandswirksam THEN
        -- alten Lagerort um alten Abgang erhöhen
        UPDATE lag
          SET lg_anztot=lg_anztot+old.l_abgg_uf1
         WHERE lg_id = TLager.lag__lg_id__from__lifsch(old);
        IF NOT found THEN
            --Lagerort gibts nicht, dann anlegen!
            PERFORM TLager.lag__from__lifsch__create(old);
        END IF;

        -- ist Abzug des neuen Abgangs möglich, abhängig von Einstellung: Negative Lagerbestände erlauben
        IF     TSystem.Settings__GetBool('no_neg_lag')
           AND coalesce( ((TLager.lag__from__lifsch(new)).lg_anztot - new.l_abgg_uf1 ), -1 ) < 0
        THEN
            PERFORM PRODAT_TEXT(lang_text(16134) || coalesce(new.l_lgort, '')); -- Achtung! Inventur nötig. Korrektur unterschreitet 0 bzw. Lagerort existiert nicht mehr (nachfolgende Buchungen vorhanden). Lagerort:
            lifschbestandswirksam:=FALSE; -- Damit in die Lagerlog ein ! kommt.
        END IF;

        -- neuen Lagerort um neuen Abgang verringern
        UPDATE lag
          SET lg_anztot=lg_anztot-new.l_abgg_uf1
         WHERE lg_id = TLager.lag__lg_id__from__lifsch(new);

        IF NOT found AND NOT TSystem.Settings__GetBool('no_neg_lag') THEN
            --alter Lagerort nicht mehr vorhanden, dann anlegen!
            --nur wirksam wenn negative Bestände erlaubt, weil sonst sowieso unter 0
            PERFORM TLager.lag__from__lifsch__create(new, true);
        END IF;
    ELSE
        PERFORM PRODAT_TEXT(lang_text(16134) || coalesce(new.l_lgort, '')); -- Achtung! Inventur nötig. Korrektur unterschreitet 0 bzw. Lagerort existiert nicht mehr (nachfolgende Buchungen vorhanden). Lagerort:
    END IF;

    PERFORM TSystem.Settings__Set('lagerlog_'||current_user, 'T');

    PERFORM TLager.lagerlog__create('=-'||IFTHEN(lifschbestandswirksam, '', ' !'), new, old); --- #14796 lifsch__a_u__lagchange()

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__a_u__lagchange
    AFTER UPDATE
    OF l_abgg, l_abgg_uf1, l_aknr, l_lgort, l_lgchnr, l_dim1, l_dim2, l_dim3, l_abg_mec
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__a_u__lagchange();
--

--Wenn jemand die Projektnummer im Lagerabgang reinschreibt, schleifen wir die auf den Lieferschein durch.
CREATE OR REPLACE FUNCTION lifsch__a_u__l_an_nr() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM disablemodified();
    UPDATE belegpos SET belp_projektnummer = new.l_an_nr
      WHERE belp_id = new.l_belp_id AND belp_belegtyp = 'LFS' AND belp_projektnummer IS DISTINCT FROM new.l_an_nr;

    UPDATE belzeil_auftg_lif SET bz_an_nr=new.l_an_nr
    FROM belegpos
    WHERE belp_belegtyp = 'LFS'
      AND belp_id       = new.l_belp_id
      AND bz_belp_id    = belp_id
      AND bz_an_nr IS DISTINCT FROM new.l_an_nr;
    PERFORM enablemodified();
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__a_u__l_an_nr
    AFTER UPDATE
    OF l_an_nr
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__a_u__l_an_nr();
--

--
CREATE OR REPLACE FUNCTION lifsch__a_d() RETURNS TRIGGER AS $$
  DECLARE rows INTEGER;
  BEGIN
    IF old.l_ag_id IS NOT NULL AND old.l_bz_id IS NULL THEN -- LA ist aus Auftrag und keiner Rechnung (auf Konsi-LFS) zugeordnet.
        PERFORM disablemodified();
        PERFORM disableauftgtrigger();
        UPDATE auftg SET ag_stkl= ag_stkl - old.l_abgg_uf1 WHERE ag_id = old.l_ag_id;
        PERFORM enableauftgtrigger();
        PERFORM enablemodified();
    END IF;

    /******************************** HIER IST SCHLUSS, WENN ARTIKEL NICHT LAGERGEFÜHRT ODER ARBEITSPAKET ********************/
    IF (NOT TArtikel.art__ac_i__is__50(old.l_aknr)) THEN
        RETURN old;
    END IF;

    -- ACHTUNG, KONSIGNATIONSPROZESS bucht nicht aus dem Lager aus, sondern macht einen Lagerortwechsel
    -- wenn der LA aus externem Konsi-Auftrag und ohne ABK und nicht über die Rechnung kommt.
    IF EXISTS(SELECT true FROM auftg WHERE ag_id = old.l_ag_id AND ag_astat = 'E' AND 'K' IN (ag_bstat, ag_bstat1, ag_bstat2))
        AND old.l_ab_ix IS NULL -- keine ABK
        AND old.l_bz_id IS NULL -- kein Rechnungsbezug (Rechnung auf Konsi-LFS)
        AND old.l_lgort_ue IS NOT NULL -- Kommisionier- / Konsignationslagerort gefüllt
    THEN
        PERFORM PRODAT_TEXT(lang_text(30083) || COALESCE(old.l_lgort, '')); -- Achtung! Konsignations-Lagerort muss manuell korrigiert werden! Lagerort: KONSI-Adresskürzel
    END IF;

    -- Sperren, wenn Lagerabgang bereits in Lieferscheinposition überführt wurde
    IF old.l_inliefdok AND (old.l_belp_id IS NOT NULL) THEN
        RAISE EXCEPTION '%',lang_text(13206);
    END IF;

    --
    PERFORM TLager.lagerlog__create('=-', NULL, old); --- #14796 lifsch__a_d()

    --
    IF old.l_abgg_uf1 > 0 THEN
        UPDATE lag
           SET lg_anztot = lg_anztot + old.l_abgg_uf1
         WHERE lg_id = TLager.lag__lg_id__from__lifsch(old);
        --Lagerort gibts nicht, dann anlegen!
        IF NOT found THEN
            PERFORM TLager.lag__from__lifsch__create(old);
        END IF;
    END IF;
    --
    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__a_d
    AFTER DELETE
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__a_d();
--

-- KS-Pflicht bei freiem Lagerabgang, #6911
CREATE OR REPLACE FUNCTION lifsch__b_iu__l_ks_abt() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (new.l_ks_abt IS NULL AND new.l_ag_id IS NULL) -- keine KS angg. und kein auftragsbezogener LA (also frei)
    IF TSystem.Settings__GetBool('KS.Pflicht.Lagab.frei') THEN
        RAISE EXCEPTION '%', lang_text(16565) || E'\n\n' || lang_text(12313) || ': ' || new.l_nr;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__b_iu__l_ks_abt
    BEFORE INSERT OR UPDATE
    OF l_ks_abt, l_ag_id
    ON lifsch
    FOR EACH ROW
    WHEN (new.l_ks_abt IS NULL AND new.l_ag_id IS NULL AND new.l_qab IS NULL)
    EXECUTE PROCEDURE lifsch__b_iu__l_ks_abt();
--

CREATE OR REPLACE FUNCTION tlager.ksv__art__chargen__restmenge_kommission__lgort___by__ab_ix__get( _ab_ix integer, _ak_nr varchar)
RETURNS varchar AS $$
DECLARE
  _smd varchar;
  _smd_bool boolean;
  _ksabt varchar;
  _ksap varchar;
BEGIN

  -- #18078,#18133 legt fest, ob ein Artikel auf den Kommissionierlagerort gelagert werden kann und gibt dann den betroffenen Lagerplatz zurück;
  -- dazu muss sowohl bei der Kostenstelle, als auch beim Artikel die passende Eigenschaft gesetzt sein.
  -- Format des Namens des Lagerortes:
  --   KOMMISSION_<KS>_<AP>
  -- bzw.
  --   KOMMISSION_<KS>_*
  -- wenn kein AP vorliegt;
  -- Gibt es keine passende Kostenstelle, dann wird null zurückgegeben.

  -- Ist für den beroffenen Artikel nicht die entsprechende recnokeyword-Eigenschaft art.lager.restmenge gesetzt,
  --   dann gibt es nichts zu tun.
  _smd_bool := false;

  SELECT TRecnoParam.GetValue( 'art.lager.restmenge', art.dbrid ) INTO _smd
  FROM art
  WHERE ak_nr = _ak_nr;

  IF isBoolean( _smd ) THEN
    _smd_bool := coalesce( _smd::boolean, false);
  END IF;

  IF NOT _smd_bool THEN
    RETURN null;
  END IF;

  -- Sucht in allen Kostenstellen der ABK nach der mit der passenden recnokeyword-Eigenschaft,
  --   und gibt Kostenstelle und Arbeitsplatz für den Arbeitsgang mit der niedrigsten Nummer zurück.
  SELECT
    ks_abt,
    coalesce( a2_ksap, '*' )
    INTO _ksabt, _ksap
  FROM ab2
  JOIN ksv ON ks_abt = a2_ks
  JOIN TRecnoParam.GetValue( 'ksv.lager.restmenge', ksv.dbrid ) AS rec ON true
  WHERE
      a2_ab_ix = _ab_ix
  AND isBoolean( rec ) AND rec::boolean
  ORDER BY a2_n
  LIMIT 1;

  RETURN 'KOMMISSION_' || _ksabt || '_' || _ksap;

END $$ LANGUAGE plpgsql STABLE STRICT;
--

CREATE OR REPLACE FUNCTION lifsch__a_i__restmenge_kommission__create() RETURNS TRIGGER AS $$
DECLARE
  _old_lgort varchar;
  _new_lgort varchar;
  _anz numeric;
  _lgid integer;
BEGIN

  -- bucht ggf. die Restmenge eines Lagerorts auf den Kommissionierlagerort einer Kostenstelle um

  -- Gibt es einen geeigneten Kommisionierlagerort?
  _new_lgort := tlager.ksv__art__chargen__restmenge_kommission__lgort___by__ab_ix__get( new.l_ab_ix, new.l_aknr );

  -- Falls ja, dann den Überschuss dorthin umbuchen.
  IF _new_lgort IS NOT null THEN

    -- Ermittlung des Überschusses, also der Lagermenge, die nicht für die ABK gebraucht wird
    SELECT lg_ort, lg_anztot INTO _old_lgort, _anz FROM lag
    WHERE
            lg_aknr = new.l_aknr
        AND lg_ort = new.l_lgort
        AND lg_chnr = new.l_lgchnr;

    -- den Überschuss auf Kommissionierlagerort umbuchen
    IF _anz > 0 THEN

        _lgid := tlager.lag__ort_chnr__lagerortwechsel
        (
          new.l_aknr,
          _old_lgort,
          new.l_lgchnr,
          _new_lgort,
          new.l_lgchnr,
          _anz
        );

      -- ABK als Bemerkung am Lagerort ergänzen
      UPDATE lag SET lg_txt = new.l_ab_ix WHERE lg_id = _lgid;

    END IF;
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;
--

CREATE TRIGGER lifsch__a_i__restmenge_kommission__create
  AFTER INSERT
  ON lifsch
  FOR EACH ROW
  WHEN ( new.l_ab_ix IS NOT null )
  EXECUTE PROCEDURE lifsch__a_i__restmenge_kommission__create();
--

CREATE OR REPLACE FUNCTION picndocu_dbrid__lifsch__chargenzeugnisse__get( _lifsch lifsch ) RETURNS varchar[] AS $$

    -- Gibt alle dbrids der Chargenzeugnisse zum Artikel des Lagerabgangs zurück.
    -- heißt: alle Wareneingangsdokumente "'quality_we'" zu Artikel und Charge

    -- Das Dokument muss mit Charge (r1 + rtemp) und Artikel (r2) verknüpft sein.
    SELECT array_agg( picndoku.dbrid )
    FROM picndoku
    JOIN recnokeyword r1 ON
              r1.r_tablename = 'picndoku'
          AND r1.r_kategorie = 'chnr'
--          AND r1.r_descr = _lifsch.l_lgchnr
-- Das sieht krank aus, beeinflusst die Performance aber positiv um mehr als das 50-fache.
-- Indizes bringen an dieser Stelle leider nichts, wie das Experiment gezeigt hat.
    JOIN recnokeyword rtemp ON rtemp.r_id = r1.r_id AND rtemp.r_descr = _lifsch.l_lgchnr
    JOIN recnokeyword r2 ON
              r2.r_tablename = 'picndoku'
          AND r2.r_kategorie = 'art'
          AND r2.r_descr = _lifsch.l_aknr
          AND r2.r_dbrid = r1.r_dbrid
    WHERE picndoku.dbrid = r1.r_dbrid
      AND pd_doktype IN ( 'quality_we', 'quality_weaw' );

  $$ LANGUAGE sql STABLE STRICT;
--


CREATE OR REPLACE FUNCTION lifsch__a_iud__picndoku() RETURNS TRIGGER AS $$
  DECLARE
    _picndoku_dbrid varchar;
    _picndoku_dbrids varchar[];
  BEGIN

    -- Fügt Chargenzeugnisse dem Lagerabgang und u.U. auch der zugehörigen ABK hinzu bzw. löscht diese, falls Artikel und Charge passen.
    -- Initial sollten Chargenzeugnisse mit Artikel, Charge und Wareneingang verknüpft sein.
    -- Das Hinzufügen dieser Verknüpfungen bedeutet aus Anwendersicht zweierlei:
    --   1. Das Chargenzeugnis ist nun im Dokumentenbaum der ABK zu finden.
    --   2. In der Dokumentenverwaltung ist das Chargenzeugnis mit LA und ABK verschlagwortet.
    -- Im Update-Fall werden zunächst die bestehenden Verknüpfungen (LA und ggf. ABK) gelöscht,
    --   um anschließend die zum geänderten Lagerabgang passenden hinzuzufügen (LA und ABK).

    -- Verknüpfung wird gelöscht, wenn LA gelöscht wird.
    IF TG_OP IN ('UPDATE', 'DELETE') THEN

        -- verknüpfte Chargendokumente, welche passende Chargen- und Artikelnummer haben
        _picndoku_dbrids := picndocu_dbrid__lifsch__chargenzeugnisse__get( old );

        DELETE FROM recnokeyword
        WHERE
                r_tablename = 'picndoku'
            AND r_kategorie = 'lifsch'
            AND r_descr = old.l_nr
            AND r_dbrid = ANY( _picndoku_dbrids );

      -- Löschung der Verknüpfung ABK Index nur, wenn es nicht noch einen anderen LA dieser ABK für die Charge/das Zeugnis gibt
      IF NOT EXISTS (
            SELECT 1 FROM lifsch
            WHERE
                    l_ag_id = old.l_ag_id
                AND l_lgchnr = old.l_lgchnr
                AND l_aknr = old.l_aknr
                AND l_ab_ix = old.l_ab_ix
                AND old.l_abgg > 0
                AND l_nr <> old.l_nr
          )
      THEN
        -- Löschung der Verknüpfungen (ABK)
        DELETE FROM recnokeyword
        WHERE
                r_tablename = 'picndoku'
            AND r_kategorie = 'abk'
            AND r_descr = old.l_ab_ix
            AND r_dbrid = ANY( _picndoku_dbrids );

      END IF;
    END IF;

    -- Hinzufügen der Verknüpfung nur, wenn LA-Menge > 0
    IF TG_OP <> 'DELETE' THEN
      IF new.l_abgg > 0 THEN

        -- Hinzufügen der Verknüpfungen + Verschlagwortung an Zeugnisse, die an der ausgebuchten Charge hängen

        _picndoku_dbrids := picndocu_dbrid__lifsch__chargenzeugnisse__get( new );
        IF _picndoku_dbrids IS NOT null THEN
          FOREACH _picndoku_dbrid IN ARRAY _picndoku_dbrids LOOP

            PERFORM CreateRecNoKeyword( 'lifsch', new.l_nr, _picndoku_dbrid );

            IF new.l_ab_ix IS NOT null THEN
              PERFORM CreateRecNoKeyword( 'abk', new.l_ab_ix, _picndoku_dbrid );
            END IF;

          END LOOP;
        END IF;

        RETURN new;
      END IF;

    END IF;

    RETURN old;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifsch__a_iud__picndoku
    AFTER INSERT OR DELETE OR UPDATE OF l_lgchnr, l_lgort, l_aknr
    ON lifsch
    FOR EACH ROW
    EXECUTE PROCEDURE lifsch__a_iud__picndoku();
--

-- Lieferschein - Verpackungsmittel
CREATE TABLE lifschpack
  (lp_id                SERIAL NOT NULL PRIMARY KEY,
   lp_l_dokunr          VARCHAR(30) NOT NULL,           --Beibehalten bei Lieferscheinumsetzung, da steht jetzt die Belegdokument-nr. drinn.
   lp_aknr              VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE,
   lp_st                numeric,        -- Anzahl der verwendeten Verpackungsmittel
   lp_buch              numeric,        -- Anzahl der verbuchten Verpackungsmittel
   lp_dim               varchar(50),    -- Geometrie der Verpackungsmittel
   lp_gewicht           numeric(12,4),  -- Gewicht der Verpackungsposition
   lp_bemerkung         text            -- Freitext
  );


CREATE OR REPLACE FUNCTION lifschpack__b_iu__lp_aknr__lp_st() RETURNS TRIGGER AS $$
  BEGIN
    -- Verpackungsgewicht wird per Standard aus dem Artikelstamm übernommen

    IF
           TG_OP = 'INSERT'
        OR (
                  TG_OP = 'UPDATE'
              AND (
                          old.lp_aknr <> new.lp_aknr
                       OR old.lp_st IS DISTINCT FROM new.lp_st
                  )

            )
    THEN
        SELECT
                coalesce( ak_gewicht, 0 ) * coalesce( new.lp_st, 1 ), ak_dim
          INTO  new.lp_gewicht                                      , new.lp_dim
        FROM art
        WHERE art.ak_nr = new.lp_aknr;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lifschpack__b_iu__lp_aknr__lp_st
    BEFORE INSERT OR UPDATE
    OF lp_aknr, lp_st
    ON lifschpack
    FOR EACH ROW
    EXECUTE PROCEDURE lifschpack__b_iu__lp_aknr__lp_st();
  ---

-- Lieferscheinkommission, fasst "gebündelte" Lagerabgänge durch Artikelbez.Lagerabgang zusammen => Übersichtsliste
CREATE TABLE lifschkomm
  (lk_kommnr            integer,        --interne Kommissionierungs-Nr. - autom. Vergabe durch Oberflächenmodul
   lk_lnr               integer NOT NULL REFERENCES lifsch ON UPDATE CASCADE ON DELETE CASCADE --zugehörige Lieferscheinnummern
  );

--
CREATE TABLE lagsernr (
  lgs_id               serial PRIMARY KEY,
  lgs_lg_id            integer REFERENCES lag    ON UPDATE CASCADE ON DELETE SET null,  -- liegt derzeit auf Lagerort
  lgs_w_wen            integer REFERENCES wendat ON UPDATE CASCADE,                     -- Wareneingangsnummer
  lgs_l_nr             integer REFERENCES lifsch ON UPDATE CASCADE,                     -- beim Lagerabgang mit Seriennummer
  lgs_sernr            varchar(75) NOT NULL,                                            -- Seriennummer
  lgs_psernr           varchar(75),                                                     -- Seriennummer aus dem die Eigenschaft ausgebaut wurde

  CONSTRAINT lagsernr__wen__lgid__lnr CHECK (    -- die Seriennummer ist ausgebucht, somit ist sie nicht mehr im lager (lgs_lg_id null)
                                                 (lgs_l_nr IS NOT null AND lgs_lg_id IS null)
                                                 -- die Seriennummer ist im Lager (Wareneingang gesetzt) dann muss auch ein Lagerortbezug da sein
                                              OR (lgs_w_wen IS NOT null AND lgs_lg_id IS NOT null)
                                                 -- die Seriennummer ist frei eingetragen (Vorgabeseriennummer) und kann noch zugewiesen werden
                                              OR (lgs_w_wen IS null)
                                             )
);

-- Indizes
    CREATE INDEX lagsernr_lgs_w_wen     ON lagsernr (lgs_w_wen) WHERE lgs_w_wen IS NOT null;
    CREATE INDEX lagsernr_lgs_l_nr      ON lagsernr (lgs_l_nr) WHERE lgs_l_nr IS NOT null;
    CREATE INDEX lagsernr_lgs_lg_id     ON lagsernr (lgs_lg_id) WHERE lgs_lg_id IS NOT null;
    CREATE INDEX lagsernr_lgs_sernr     ON lagsernr (lgs_sernr);
--

-- prüft, ob die übergebene Seriennummer mit einer bereits vorhandenen kollidiert
-- Rückgabe true: neue Seriennummer bislang unbekannt oder keine Kollision mit vorhandener
-- Rückgabe false: Kollision kann behoben werden, indem man die neue Seriennummer nicht einfügt
--    ( neue Seriennummer ist eine Vorgabeseriennummer und Bestellposition/Seriennummer-Kombination ist schon vorhanden )
-- Exception: kritische Kollision ( Artikel/Charge/Seriennummer-Kombination bereits im Lager )
CREATE OR REPLACE FUNCTION tlager.lagsernr__sernr__duplikate__check( _lagsernr lagsernr ) RETURNS boolean AS $$
DECLARE
  _lgs_ids_vorhanden integer[];
  _lg_aknr varchar;
  _lg_chnr varchar;
  _ld_id varchar;
  _vorgabesernr_vorhanden boolean;
BEGIN

  -- vorhandene Seriennummern gleichen Namens auslesen
  _lgs_ids_vorhanden :=
    array_agg( lgs_id ) FROM lagsernr lgs
    WHERE lgs.lgs_sernr = _lagsernr.lgs_sernr AND lgs.lgs_id <> _lagsernr.lgs_id;

  IF _lgs_ids_vorhanden IS NOT null THEN

    -- Exception werfen, falls gleiche Artikel/Charge/Seriennummer-Kombination mit Lagerbestand vorhanden
    SELECT lg_aknr, lg_chnr INTO _lg_aknr, _lg_chnr FROM lag WHERE lg_id = _lagsernr.lgs_lg_id AND lg_anztot > 0;

    IF EXISTS(
      SELECT 1 FROM lag
      JOIN lagsernr ON lgs_lg_id = lg_id AND lgs_id = ANY( _lgs_ids_vorhanden )
      WHERE lg_aknr = _lg_aknr AND lg_chnr = _lg_chnr AND lg_anztot > 0
    ) THEN
      RAISE EXCEPTION 'Gleiche Seriennummern sind nicht zulässig. xtt26310 %', _lagsernr.lgs_sernr;
    END IF;

    -- false zurückgeben, wenn Vorgabebestellposition/Seriennummer-Kombination bereits vorhanden, ansonsten kein Konflikt
    _ld_id := ms_pkey FROM mapsernr WHERE ms_table = 'ldsdok'::REGCLASS AND ms_lgs_id = _lagsernr.lgs_id;
    _vorgabesernr_vorhanden := EXISTS(
      SELECT 1 FROM mapsernr WHERE ms_table = 'ldsdok'::REGCLASS AND ms_pkey = _ld_id AND ms_lgs_id = ANY( _lgs_ids_vorhanden )
    );

    RETURN NOT _vorgabesernr_vorhanden;
  END IF;

  RETURN true;

END $$ LANGUAGE plpgsql STRICT VOLATILE;

CREATE OR REPLACE FUNCTION lagsernr__a_iu__duplikat__check() RETURNS TRIGGER AS $$
  BEGIN

    -- prüft, ob hinzugefügte Seriennummer mit einer bestehenden Seriennummer kollidiert
    --  und löscht die Doublette gegebenenfalls
    IF NOT tlager.lagsernr__sernr__duplikate__check( new ) THEN
      DELETE FROM lagsernr WHERE lgs_id = new.lgs_id;
    END IF;

    RETURN null;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagsernr__a_iu__duplikat__check
    AFTER INSERT OR UPDATE OF lgs_lg_id, lgs_sernr
    ON lagsernr
    FOR EACH ROW
    EXECUTE PROCEDURE lagsernr__a_iu__duplikat__check();
--

--
CREATE OR REPLACE FUNCTION lagsernr__b_iu() RETURNS TRIGGER AS $$
  DECLARE
    _lgs_id__vorgabe integer;
    _w_wen__vorgabe integer;
    _ld_id__vorgabe integer;
  BEGIN
    IF new.lgs_l_nr IS NOT null THEN  --Ausgebucht, hängt nicht mehr am Lagerort!
      new.lgs_lg_id := null;
    END IF;

    new.lgs_sernr := trim( new.lgs_sernr );

    -- Die Seriennummer wird einem Lagerzugang zugewiesen. Somit muss sie auch mit dem Lager-Datensatz verknüpft werden, da sonst Inventur usw. nicht funktionieren
    IF     (new.lgs_w_wen IS NOT null AND new.lgs_lg_id IS null)
           -- wenn ich das nachträglich eintrage und auch sofort die ausbuchung dazu, kann es kein lagerort mehr geben da die Seriennummer bereits weg ist.
           -- dann hätte l_nr einen Wert
       AND (new.lgs_l_nr IS null)
    THEN
       new.lgs_lg_id := TLager.lag__lg_id__from__wendat(wendat) FROM wendat WHERE w_wen = new.lgs_w_wen;
    END IF;

    -- Die Seriennummer wird einem Lagerort zugewiesen (permanente Inventur). Wenn der Lagerort einem Lagerzugang zugewiesen ist, dann kann auch die SN dem Lagerzugang zugewiesen werden.
    IF ( new.lgs_w_wen IS null AND new.lgs_lg_id IS NOT null ) THEN
      new.lgs_w_wen := lg_w_wen FROM lag WHERE lg_id = new.lgs_lg_id;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagsernr__b_iu
    BEFORE INSERT OR UPDATE
    ON lagsernr
    FOR EACH ROW
    EXECUTE PROCEDURE lagsernr__b_iu();


  CREATE OR REPLACE FUNCTION lagsernr__a_iu__l_nr__lg_id() RETURNS TRIGGER
    AS $$
    BEGIN
        -- triggert lag__a_iu wo leere lagerorte gelöscht werden. das müßte eigentlich in eine funktion (leere lagerorte löschen)
        UPDATE lag SET lg_anztot = lg_anztot WHERE lg_id = old.lgs_lg_id AND lg_anztot <= 0;
        RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER lagsernr__a_iu__l_nr__lg_id
      AFTER UPDATE OF lgs_lg_id, lgs_l_nr
      ON lagsernr
      FOR EACH ROW
      EXECUTE PROCEDURE lagsernr__a_iu__l_nr__lg_id();
--

-- Trigger löscht Lagsernummer, wenn keine Bezüge mehr vorhanden sind
  CREATE OR REPLACE FUNCTION lagsernr__a_01_u__check_orphan() RETURNS TRIGGER AS $$
    BEGIN
      IF NOT EXISTS(SELECT true FROM mapsernr WHERE ms_lgs_id = new.lgs_id) THEN
        DELETE FROM lagsernr WHERE lgs_id = new.lgs_id;
        RETURN NULL;
      ELSE
        RETURN new;
      END IF;
    END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagsernr__a_01_u__check_orphan
    AFTER UPDATE OF lgs_lg_id, lgs_l_nr, lgs_w_wen
    ON lagsernr
    FOR EACH ROW
      WHEN ((new.lgs_w_wen IS NULL) AND (new.lgs_lg_id IS NULL) AND (new.lgs_l_nr IS NULL))
    EXECUTE PROCEDURE lagsernr__a_01_u__check_orphan();

  -- nach dem LA von Seriennummern das löschen des LO nochmal anstoßen (lag__a_iu), falls notwendig
  -- https://redmine.prodat-sql.de/issues/16515
  CREATE OR REPLACE FUNCTION lagsernr__a_02_ud__delete_from_lag__if__empty() RETURNS TRIGGER AS $$
    BEGIN
       IF    (tg_op = 'DELETE')
          OR (    (tg_op = 'UPDATE')
              AND (new.lgs_lg_id IS NULL)     -- Lagerortbezug entfernt. Passiert bei Lagerabgang, siehe lagsernr__b_iu
              AND (new.lgs_l_nr IS NOT NULL)) -- wenn Lagerabgang gesetzt ist
       THEN
         UPDATE lag SET lg_id = lg_id WHERE lg_id = old.lgs_lg_id AND lg_anztot <= 0;
       END IF;

       IF (tg_op = 'DELETE') THEN
         RETURN old;
       END IF;

       RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER lagsernr__a_02_ud__delete_from_lag__if__empty
      AFTER DELETE OR UPDATE
      OF lgs_l_nr                             -- wenn Lagerabgang als Auslöser
      ON lagsernr
      FOR EACH ROW
      EXECUTE PROCEDURE lagsernr__a_02_ud__delete_from_lag__if__empty();
  --
--

--
/*
CREATE OR REPLACE FUNCTION lagsernr__a_i() RETURNS TRIGGER AS $$
  BEGIN
    INSERT INTO lagartsernrkonf (lak_lgs_id, lak_aknr, lak_ep, lak_txt) SELECT new.lgs_id, ldak_aknr, ldak_ep, ldak_txt FROM ldsartkonf JOIN ldsdok ON ldak_ld_id=ld_id JOIN wendat ON ld_id=w_lds_id WHERE w_wen=new.lgs_w_wen;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagsernr__a_i
    AFTER INSERT
    ON lagsernr
    FOR EACH ROW
    EXECUTE PROCEDURE lagsernr__a_i();
*/
--

--
/*
CREATE OR REPLACE FUNCTION lagsernr__a_iu() RETURNS TRIGGER AS $$
  DECLARE r RECORD;
  BEGIN
    IF TSystem.Settings__Get('KUNDE') IN ('SUNSTROM','MR-SUN') THEN
      PERFORM prepare_wrverwaltung();
      IF NOT EXISTS(SELECT true FROM sun_wechselrichter WHERE wr_sernr=new.lgs_sernr) THEN
        INSERT INTO sun_wechselrichter(wr_sernr) VALUES (new.lgs_sernr);
      END IF;
      UPDATE sun_wechselrichter SET wr_we_nr=new.lgs_w_wen, wr_l_nr=new.lgs_l_nr WHERE wr_sernr=new.lgs_sernr;

      FOR r IN SELECT lak_aknr FROM lagartsernrkonf WHERE lak_lgs_id=new.lgs_id LOOP
        EXECUTE 'UPDATE sun_wechselrichter SET '||quote_ident(r.lak_aknr)||'='||quote_literal(new.lgs_sernr)||' WHERE wr_sernr='||quote_literal(new.lgs_sernr);
      END LOOP;
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagsernr__a_iu
    AFTER INSERT OR UPDATE
    ON lagsernr
    FOR EACH ROW
    EXECUTE PROCEDURE lagsernr__a_iu();
*/
--
/*
 CREATE TABLE lagartsernrkonf
  (lak_id               SERIAL PRIMARY KEY,
   lak_lgs_id           INTEGER NOT NULL REFERENCES lagsernr ON DELETE CASCADE,
   lak_aknr             VARCHAR(40) NOT NULL,
   lak_ep               NUMERIC(12,4),
   lak_txt              TEXT
  );

 CREATE INDEX lagartsernrkonf_lak_aknr ON lagartsernrkonf(lak_aknr);


 CREATE OR REPLACE FUNCTION lagartsernrkonf__a_d() RETURNS TRIGGER AS $$
 DECLARE r RECORD;
 BEGIN
  IF TSystem.Settings__Get('KUNDE') IN ('SUNSTROM','MR-SUN') THEN
        --RAISE EXCEPTION '%', (SELECT lgs_sernr FROM lagsernr WHERE lgs_id=old.lak_lgs_id);
        EXECUTE 'UPDATE sun_wechselrichter SET '||quote_ident(old.lak_aknr)||'=NULL WHERE wr_sernr='||quote_literal((SELECT lgs_sernr FROM lagsernr WHERE lgs_id=old.lak_lgs_id));
  END IF;
  --
  RETURN new;
 END$$LANGUAGE plpgsql;

 CREATE TRIGGER lagartsernrkonf__a_d
 AFTER DELETE
 ON lagartsernrkonf
 FOR EACH ROW
 EXECUTE PROCEDURE lagartsernrkonf__a_d();
*/
--

-- Zuordnung der Seriennummer zu verschiedenen Vorgängen
CREATE TABLE mapsernr(
    ms_id           SERIAL PRIMARY KEY,
    ms_lgs_id       INTEGER NOT NULL REFERENCES lagsernr ON DELETE CASCADE,     -- Master-SN
    ms_table        REGCLASS NOT null,                                          -- Detail-Tabelle
    ms_pkey         TEXT NOT NULL                                               -- Detail-ID (Primärschlüssel)
);

-- Seriennummerzuweisungen müssen bzgl. ihrer Zuweisung eindeutig sein
CREATE UNIQUE INDEX mapsernr__ms_lgs_id__ms_table__ms_pkey__uq ON mapsernr ( ms_lgs_id, ms_table, ms_pkey );
CREATE INDEX mapsernr__ms_lgs_id ON mapsernr ( ms_lgs_id );
CREATE INDEX mapsernr__ms_table__ms_pkey__uq ON mapsernr ( ms_table, ms_pkey );

CREATE OR REPLACE FUNCTION mapsernr__a_iu__duplikat__check() RETURNS TRIGGER AS $$
  DECLARE
    _lagsernr lagsernr;
  BEGIN

      -- Prüft, ob hinzugefügte Seriennummer mit einer bestehenden Seriennummer kollidiert.
      -- Kollision bedeutet an dieser Stelle:
      ---- 1. Dieselbe Seriennummer/Artikel/Chargen-Kombination mit Lagerbestand soll eingefügt werden -> Exception
      ---- 2. Dieselbe Vorgabeeriennummer/Bestellposition soll eingefügt werden -> Duplikat wird verworfen
      _lagsernr := lagsernr FROM lagsernr WHERE lgs_id = new.ms_lgs_id;

      -- Ist das der Fall, dann wird die soeben angelegte Seriennummer wieder gelöscht.
      IF NOT tlager.lagsernr__sernr__duplikate__check( _lagsernr ) THEN
        DELETE FROM lagsernr WHERE lgs_id = _lagsernr.lgs_id;
      END IF;

    RETURN null;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER mapsernr__a_iu__duplikat__check
    AFTER INSERT OR UPDATE OF ms_table, ms_pkey, ms_lgs_id
    ON mapsernr
    FOR EACH ROW
    WHEN ( new.ms_table = 'ldsdok'::REGCLASS )
    EXECUTE PROCEDURE mapsernr__a_iu__duplikat__check();
--

-- Trigger löscht Lagerseriennummer, wenn keine Bezüge mehr vorhanden sind
  CREATE OR REPLACE FUNCTION mapsernr__a_d__check_orphan() RETURNS TRIGGER AS $$
    BEGIN
     IF EXISTS(SELECT true FROM lagsernr
               WHERE (lgs_id = old.ms_lgs_id)
                 AND (lgs_w_wen IS NULL)
                 AND (lgs_lg_id IS NULL)
                 AND (lgs_l_nr IS NULL)
              )
     THEN
       IF NOT EXISTS(SELECT true FROM mapsernr WHERE ms_lgs_id = old.ms_lgs_id AND NOT ms_id = old.ms_id) THEN
         DELETE FROM lagsernr WHERE lgs_id = old.ms_lgs_id;
       END IF;
     END IF;
     RETURN old;
    END $$ LANGUAGE plpgsql;

  CREATE TRIGGER mapsernr__a_d__check_orphan
    AFTER DELETE
    ON mapsernr
    FOR EACH ROW
    EXECUTE PROCEDURE mapsernr__a_d__check_orphan();
--

CREATE OR REPLACE FUNCTION mapsernr__b_iu__ldsdok() RETURNS TRIGGER AS $$
  BEGIN
    -- Trigger prüft, ob Bestellung tatsächlich existiert

    IF NOT EXISTS( SELECT 1 FROM ldsdok WHERE ld_id = new.ms_pkey ) THEN
      RAISE EXCEPTION 'Bestellung wurde nicht gefunden. xtt10497 %', new.ms_pkey;
    END IF;

    RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER mapsernr__b_iu__ldsdok
    BEFORE INSERT OR UPDATE
    ON mapsernr
    FOR EACH ROW
    WHEN ( new.ms_table = 'ldsdok'::REGCLASS )
    EXECUTE PROCEDURE mapsernr__b_iu__ldsdok();
--

CREATE TABLE lagerlog (
  lo_id                serial PRIMARY KEY,
  lo_datum             timestamp(0) without time zone DEFAULT now(),
  lo_user              varchar(100) DEFAULT tsystem.current_user_ll_db_usename(true),
  lo_aktion            varchar(5),  -- - =- =-! ++ + =+ =+! == ==- ==+ ?=I --I ++I ==I --2 K
  lo_wendat            integer, -- kein REFERENCES da sonst löschen von Lagerbewegungen nicht geht
  lo_lifsch            integer, -- kein REFERENCES da sonst löschen von Lagerbewegungen nicht geht
  lo_aknr_old          varchar(40),
  lo_aknr_new          varchar(40),
  lo_stk_old           numeric,
  lo_stk_new           numeric,
  lo_dim1_old          numeric,
  lo_dim1_new          numeric,
  lo_dim2_old          numeric,
  lo_dim2_new          numeric,
  lo_dim3_old          numeric,
  lo_dim3_new          numeric,
  lo_lgort_old         varchar(50),
  lo_lgort_new         varchar(50),
  lo_lgchnr_old        varchar(50),
  lo_lgchnr_new        varchar(50),
  lo_stat              varchar(1),
  lo_nr                varchar(40),
  lo_pos               integer,
  lo_txt               text,
  lo_bestand           numeric,
  lo_lg_bestand        numeric,
  lo_vkpbas            numeric,
  lo_hest              numeric,
  lo_adpreis           numeric, -- Durchschnittspreis siehe auch wendat; gesetzt durch BelPreisBuchen
  lo_umbuchid          integer

  -- System (tables__generate_missing_fields)
  --   kein automatisches dbrid, insert_date, insert_by, modified_by, modified_date und table_delete-Trigger (tables__fieldInfo__fetch)
  );

-- Indizes
  CREATE INDEX lagerlog_aknr_dat        ON lagerlog (lo_aknr_new, lo_datum);
  CREATE INDEX lagerlog_aknr_dat_like   ON lagerlog (lo_aknr_new varchar_pattern_ops, lo_datum);

--  #19742 Index ist mangels Verwendung überflüssig
--  CREATE INDEX lagerlog_lo_txt          ON lagerlog USING HASH ( lo_txt );
  CREATE INDEX lagerlog_lo_txt_like     ON lagerlog (lo_txt varchar_pattern_ops);
  CREATE INDEX lagerlog_ref             ON lagerlog (lo_stat, lo_nr, lo_pos);
  CREATE INDEX lagerlog_lo_nr_like      ON lagerlog (lo_nr varchar_pattern_ops);
  CREATE INDEX lagerlog_dat             ON lagerlog (timestamp_to_date(lo_datum));
  CREATE INDEX lagerlog_wendat          ON lagerlog (lo_wendat);
  CREATE INDEX lagerlog_lifsch          ON lagerlog (lo_lifsch);
  CREATE INDEX lagerlog_datum           ON lagerlog (CAST(lo_datum AS DATE));
  CREATE INDEX lagerlog_lo_umbuchid     ON lagerlog (lo_umbuchid);
  CREATE INDEX lagerlog_lagerbeweg_old  ON lagerlog (lo_aknr_new, COALESCE(lo_lgort_old, ''), COALESCE(lo_lgchnr_old, '')); -- spezieller Index für BI Lagerbewegungen
  CREATE INDEX lagerlog_lagerbeweg_new  ON lagerlog (lo_aknr_new, COALESCE(lo_lgort_new, ''), COALESCE(lo_lgchnr_new, '')); -- spezieller Index für BI Lagerbewegungen
--

--
CREATE OR REPLACE FUNCTION lagerlog__b_i() RETURNS TRIGGER AS $$
  BEGIN
    IF current_user='syncro' THEN
        RETURN new;
    END IF;
    IF current_user='APPS' THEN
        RETURN NULL;
    END IF;

    -- Warenausgang
    IF new.lo_lifsch IS NOT NULL THEN
        SELECT ag_astat, ag_nr, ag_pos
          INTO new.lo_stat, new.lo_nr, new.lo_pos
          FROM lifsch LEFT JOIN auftg ON l_ag_id = ag_id
         WHERE l_nr = new.lo_lifsch;
    END IF;

    -- Wareneingang
    IF new.lo_wendat IS NOT NULL THEN
        SELECT ld_code, ld_auftg, ld_pos
          INTO new.lo_stat, new.lo_nr, new.lo_pos
          FROM wendat LEFT JOIN ldsdok ON w_lds_id = ld_id
         WHERE w_wen = new.lo_wendat;
    END IF;

    -- Statistik für Preise mitfuehren, bei jeder Lagerbewegung wird aktualisiert
    SELECT ak_hest, ak_vkpbas  INTO new.lo_hest, new.lo_vkpbas FROM art WHERE ak_nr=new.lo_aknr_new;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerlog__b_i
    BEFORE INSERT
    ON lagerlog
    FOR EACH ROW
    EXECUTE PROCEDURE lagerlog__b_i();
--

--
CREATE OR REPLACE FUNCTION lagerlog__b_iu() RETURNS TRIGGER AS $$
  BEGIN
    IF    current_user = 'syncro'
    THEN
        RETURN new;
    END IF;

    -- Bestandsabgleich sofort durchführen, da Lagerlog-Trigger sonst die alten "Lagerbestand (ak_tot)" Werte sieht
    PERFORM tartikel.bestand_abgleich_intern( new.lo_aknr_new );

    IF new.lo_bestand IS NULL THEN
        new.lo_bestand := ak_tot FROM art  WHERE ak_nr = new.lo_aknr_new;
    END IF;

    IF new.lo_lg_bestand IS NULL THEN

        new.lo_lg_bestand :=
               lg_anztot
          FROM lag
         WHERE lg_aknr = new.lo_aknr_new
           AND lg_ort  = coalesce( new.lo_lgort_new, '' )
           AND lg_chnr = coalesce( new.lo_lgchnr_new, '' )
           AND lg_dim1 = new.lo_dim1_new
           AND lg_dim2 = new.lo_dim2_new
           AND lg_dim3 = new.lo_dim3_new;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER lagerlog__b_iu
    BEFORE INSERT OR UPDATE
    ON lagerlog
    FOR EACH ROW
    EXECUTE PROCEDURE lagerlog__b_iu();
--

-- Todo - Funktion trennen, mit und ohne Rechnung

-- DOKU
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Inventur
-- DOKU

--
CREATE OR REPLACE FUNCTION lagerlog_stichtagbestand(
      _stichtag date,
      _inaknr   varchar = null,
      _inlgort  varchar = null,
      _inakacv  varchar = null,
      _inakacb  varchar = null,
      _inakic   varchar = null,

      OUT aknr varchar,   OUT dat date,           OUT bestand numeric,
      OUT hest numeric,   OUT vkp numeric,        OUT adpreis numeric,

      OUT letzteErechnung_preis_stichtag numeric, OUT letzteErechnung_preis numeric,
      OUT letzteArechnung_preis_stichtag numeric, OUT letzteArechnung_preis numeric,
      OUT letztefertigung_preis_stichtag numeric, OUT letztefertigung_preis numeric
  ) RETURNS SETOF record AS $$
  DECLARE
      _last_lagerlog  record;
      _lagerlog_data  record;
  BEGIN
    -- letzte Lagerbewegung vor dem Stichtag hält den Lagerbestand zum Stichtag
    FOR _last_lagerlog IN
        SELECT
          lo_aknr_new,
          max( lo_datum ),
          max( lo_id ) AS maxloid
        FROM lagerlog
          JOIN art    ON ak_nr = lo_aknr_new
          JOIN artcod ON ak_ac = ac_n
        WHERE ak_lag
          AND lo_datum::date <= _stichtag
          -- Artikel einschränken
          AND CASE WHEN _inaknr IS NOT NULL THEN
                  lo_aknr_new LIKE _inaknr
              ELSE true
              END
          -- Lagerplatz einschränken
          AND CASE WHEN _inlgort IS NOT NULL THEN
                  lo_lgort_new LIKE _inlgort
              ELSE true
              END
          -- AC einschränken
          AND CASE WHEN _inakacv IS NOT NULL AND _inakacb IS NOT NULL THEN
                  ak_ac BETWEEN _inakacv AND _inakacb
              ELSE true
              END
          -- IC einschränken
          AND CASE WHEN _inakic IS NOT NULL THEN
                  ac_i LIKE _inakic
              ELSE true
              END
          -- nur werthaltige Lagerorte, PHKO #13475
          AND coalesce( (lagerorte__get_setup( lo_lgort_new ))._werthg, true )

        GROUP BY
          lo_aknr_new,
          CASE WHEN _inlgort IS NOT NULL THEN lo_lgort_new || lo_lgchnr_new ELSE null END
        ORDER BY lo_aknr_new
    LOOP
        SELECT
          *,
          tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( lo_aknr_new )
        INTO _lagerlog_data
        FROM lagerlog
        WHERE lo_id = _last_lagerlog.maxloid;

        IF  (
                  (_inlgort IS NOT NULL  AND coalesce( _lagerlog_data.lo_lg_bestand, 0 ) > 0)
              OR  (_inlgort IS NULL      AND coalesce( _lagerlog_data.lo_bestand, 0 )    > 0)
            )
        THEN
            dat   := _lagerlog_data.lo_datum;
            aknr  := _lagerlog_data.lo_aknr_new;

            -- Gesamtlagerbestand über alle Lagerorte
            IF _inlgort IS NOT NULL THEN
                bestand := coalesce( _lagerlog_data.lo_lg_bestand, 0 );
            ELSE
                bestand := coalesce( _lagerlog_data.lo_bestand, 0 );
            END IF;

            hest := _lagerlog_data.lo_hest;
            vkp  := _lagerlog_data.lo_vkpbas;

            letzteErechnung_preis_stichtag := TEingRech.letzte_einkaufsPreis_dat( aknr, tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( aknr ), _stichtag,    false, false );
            letzteErechnung_preis          := TEingRech.letzte_einkaufsPreis_dat( aknr, tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( aknr ), current_date, false, false );
            letzteArechnung_preis_stichtag := TFaktura.letzte_rechnung_dat(       aknr, tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( aknr ), _stichtag );
            letzteArechnung_preis          := TFaktura.letzte_rechnung_dat(       aknr, tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( aknr ), current_date );

            SELECT  ab_nk_et
            INTO    letztefertigung_preis_stichtag
            FROM ldsdok
              JOIN abk ON ab_ix = ld_abk
            WHERE ld_aknr = aknr
              AND ld_abk IS NOT NULL
              AND ld_term >= _stichtag - 365
              AND ld_term <= _stichtag
              AND ab_buch
              -- Nacharbeiten sollen irgnoriert werden falls es sich beim QAB/SV um Kundenreklamationen handelt #18891
              AND coalesce( ( SELECT NOT q_festext FROM qab WHERE q_nr = ldsdok.ld_q_nr ), true )
            ORDER BY ld_abk DESC
            LIMIT 1;

            SELECT  ab_nk_et
            INTO    letztefertigung_preis
            FROM ldsdok
              JOIN abk ON ab_ix = ld_abk
            WHERE ld_aknr = aknr
              AND ld_abk IS NOT NULL
              AND ld_term >= current_date - 365
              AND ld_term <= current_date
              AND ab_buch
              -- Nacharbeiten sollen irgnoriert werden falls es sich beim QAB/SV um Kundenreklamationen handelt #18891
              AND coalesce( ( SELECT NOT q_festext FROM qab WHERE q_nr = ldsdok.ld_q_nr ), true )
            ORDER BY ld_abk DESC
            LIMIT 1;

            -- adpreis := _lagerlog_data.lo_adpreis;
            -- wir suchen den letzten berechneten Durchschnittspreis zu diesem Lagerlog-Satz
            SELECT  lo_adpreis
            INTO    adpreis
            FROM lagerlog
            WHERE lo_aknr_new = _lagerlog_data.lo_aknr_new
              AND lo_id <= _lagerlog_data.lo_id
              AND lo_adpreis IS NOT NULL
            ORDER BY lo_id DESC
            LIMIT 1;

            RETURN NEXT;
        END IF;
    END LOOP;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

--
 CREATE TABLE haenel
  (hl_id                SERIAL PRIMARY KEY,
   hm_inpath            VARCHAR(100),                  -- ??
   hl_outpath           VARCHAR(100),                  -- Export-Pfad für Lift-Aufträge
   hl_outfile           VARCHAR(10) DEFAULT 'job',     -- Dateierweiterung für Lift-Aufträge
   hl_lg_ort            VARCHAR(50),                   -- Lagerort, für den die Lift-Aufträge ausgeführt werden sollen (LIKE!!) -- z.B.: "HL1 | %"
   hl_art_export       BOOLEAN DEFAULT false,         -- Artikelstammdaten zum Hänel exportieren (insert und update im Hänel)
   hl_art_outpath       VARCHAR(100),                  -- Export-Pfad für Artikelstammdaten; wenn leer, dann hl_outpath
   hl_art_outfile       VARCHAR(10) DEFAULT 'csv'      -- Dateierweiterung für Artikelstammdaten
  );


 CREATE TABLE haenel_aktion
  (ha_id                SERIAL PRIMARY KEY,
   ha_aktion            VARCHAR(1),
   ha_aknr              VARCHAR(40),
   ha_auftg             VARCHAR(50),
   ha_nr_pos            VARCHAR(20),
   ha_stk               NUMERIC,
   ha_lgort             VARCHAR(50)
  );

 CREATE OR REPLACE FUNCTION haenel_aktion__b_iu() RETURNS TRIGGER AS $$
 BEGIN
  IF NOT current_user IN (SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND groname='SYS.Lager-Haenel') THEN
        RAISE EXCEPTION '%', Format(lang_text(29175) /*'Sie müssen in der Benutzergruppe "SYS.Lager-Haenel" sein um Aufträge an den Lift schicken zu können'*/);
  END IF;
  new.ha_stk:=ROUND(new.ha_stk);
  RETURN new;
 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER haenel_aktion__b_iu
 BEFORE INSERT OR UPDATE
 ON haenel_aktion
 FOR EACH ROW
 EXECUTE PROCEDURE haenel_aktion__b_iu();


 CREATE TABLE hregal
  (hr_rnr               VARCHAR(4) NOT NULL PRIMARY KEY,
   hr_bez               VARCHAR(75)
  );


CREATE TABLE regalpos
 (rp_id                 SERIAL PRIMARY KEY,
  rp_rnr                VARCHAR(4) NOT NULL REFERENCES hregal ON UPDATE CASCADE,
  rp_lae                VARCHAR(2) NULL,
  rp_hoe                VARCHAR(2) NOT NULL,
  rp_pnr                INTEGER,
  --rp_hom              NUMERIC,
  rp_gro                INTEGER NOT NULL,
  rp_tei                INTEGER NOT NULL,
  rp_sta                INTEGER NOT NULL DEFAULT 0,
  rp_spr                BOOL NOT NULL DEFAULT FALSE
 );

CREATE OR REPLACE FUNCTION hregal_createregalpos(regalmask VARCHAR, anzlae INTEGER, anzhoe INTEGER, groe INTEGER, teil INTEGER) RETURNS BOOL AS $$
DECLARE l INTEGER;
        h INTEGER;
        r RECORD;
BEGIN
 FOR r IN SELECT hr_rnr FROM hregal WHERE hr_rnr LIKE regalmask LOOP --alle Regale, welcher der Maskierung passend sind durchlafuen
        FOR l IN 1..anzlae LOOP --alle Längspositionen
                FOR h IN 1..anzhoe LOOP --alle Höhenpositionen
                        IF NOT EXISTS(SELECT true FROM regalpos WHERE rp_rnr=r.hr_rnr AND rp_lae=lpad(l, 2, 0) AND rp_hoe=lpad(h, 2, 0)) THEN --schauen ob schon da
                                INSERT INTO regalpos(rp_rnr, rp_lae, rp_hoe, rp_gro, rp_tei, insert_by) VALUES (r.hr_rnr, lpad(l, 2, 0), lpad(h, 2, 0), groe, teil, 'auto'); --nein? dann anlegen mit größe und teilung wie angegeben
                        END IF;
                END LOOP;
        END LOOP;
 END LOOP;
 --
 RETURN true;
END $$ LANGUAGE plpgsql;

--Inventur-Kopfdaten
CREATE TABLE inv (
  iv_nr                 VARCHAR(30) PRIMARY KEY,              --Inventur-Nummer
  iv_bdat               DATE DEFAULT current_date,            --Inventur-Beginn
  iv_edat               DATE,                                 --Inventur-Ende
  iv_bdat_time          TIME(0) WITHOUT TIME ZONE DEFAULT current_time,  --Inventur-Beginn mit Zeitstempel
  iv_edat_time          TIME(0) WITHOUT TIME ZONE ,
  iv_txt                TEXT,                                 --Zusatztext
  iv_txt_rtf            TEXT,
  iv_def                BOOLEAN NOT NULL DEFAULT FALSE,       --Inventur wurde abschlossen/zurückgeschrieben
  -- Konsistenz von Datum und Zeit für Anfang und Ende sicherstellen, #13800
  CONSTRAINT inv_chk_iv_bdat_iv_bdat_time CHECK ((iv_bdat IS NULL AND iv_bdat_time IS NULL) OR (iv_bdat IS NOT NULL AND iv_bdat_time IS NOT NULL)),
  CONSTRAINT inv_chk_iv_edat_iv_edat_time CHECK ((iv_edat IS NULL AND iv_edat_time IS NULL) OR (iv_edat IS NOT NULL AND iv_edat_time IS NOT NULL))
 );
--

CREATE OR REPLACE FUNCTION inv__b_u() RETURNS TRIGGER AS $$
 BEGIN

  IF new.iv_def AND NOT old.iv_def THEN
    new.iv_edat := COALESCE(new.iv_edat,current_date);
  END IF;

  RETURN new;
 END $$ LANGUAGE plpgsql;
--

CREATE TRIGGER inv__b_u
  BEFORE UPDATE
  ON inv
  FOR EACH ROW
 EXECUTE PROCEDURE inv__b_u();
--

--
CREATE OR REPLACE FUNCTION inv__b_iu__date_time() RETURNS TRIGGER AS $$
  -- Konsistenz von Datum und Zeit für Anfang und Ende sicherstellen, #13800
  -- inv.iv_bdat und inv.iv_bdat_time sind entweder beide null oder beide nicht null
  -- analog für inv.iv_edat und inv.iv_edat_time
  BEGIN
    IF new.iv_bdat IS NULL THEN
        new.iv_bdat_time := NULL;
    END IF;

    IF new.iv_bdat IS NOT NULL AND new.iv_bdat_time IS NULL THEN
        new.iv_bdat_time := '00:00'::TIME;
    END IF;

    IF new.iv_edat IS NULL THEN
        new.iv_edat_time := NULL;
    END IF;

    IF new.iv_edat IS NOT NULL AND new.iv_edat_time IS NULL THEN
        new.iv_edat_time := '00:00'::TIME;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER inv__b_iu__date_time
    BEFORE INSERT OR UPDATE
    OF iv_bdat, iv_bdat_time, iv_edat, iv_edat_time
    ON inv
    FOR EACH ROW
    EXECUTE PROCEDURE inv__b_iu__date_time();
--

--Inventur-Bestände
CREATE TABLE invlag (
  il_id                 SERIAL NOT NULL PRIMARY KEY,
  il_iv_nr              VARCHAR(30) NOT NULL REFERENCES inv ON DELETE CASCADE ON UPDATE CASCADE,        -- Für welche Inventur
  il_aknr               VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE,                          -- Welcher Artikel
  il_lg_id              INTEGER,                                                -- Lagerort für den Inventur stattfindet, NULL bei neu eingefügten Lagerorten
  il_txt                TEXT,                                                   -- Zusatztext zu Inv.Position
  il_txt_rtf            TEXT,
  il_ort                VARCHAR(50) NOT NULL DEFAULT '',                        -- Felder analog Lageraufbau
  il_anztot             NUMERIC(12,2) NOT NULL DEFAULT 0,                       -- Inventurbestand
  il_anzist             NUMERIC(12,2) NOT NULL DEFAULT 0,                       -- Lagermenge
  il_mce                INTEGER REFERENCES artmgc ON DELETE SET NULL,
  il_zaedat             DATE,                                                   -- Datum Erfassung Inventurbestand
  il_konsDat            DATE,                                                   -- Konservierungsdatum (LOLL : Nutzer tragen manuell ein, wann Bauteile konserviert (Korrosionsschutz) wurden)
  il_konsEnde           DATE,                                                   -- Ablauf der Konservierung
  il_chnr               VARCHAR(50) NOT NULL DEFAULT '',
  il_dim1               NUMERIC(12,4) NOT NULL DEFAULT 0,
  il_dim2               NUMERIC(12,4) NOT NULL DEFAULT 0,
  il_dim3               NUMERIC(12,4) NOT NULL DEFAULT 0,
  il_sperr              BOOLEAN DEFAULT FALSE,
  il_regalsta           INTEGER,
  il_delete             BOOLEAN NOT NULL DEFAULT FALSE,                         -- Lagerort soll entfernt werden beim Zurückschreiben
  il_ak_vkpbas          NUMERIC(12,4),                                          -- Verkaufspreisbasis zum Zeitpunkt der Inventuranlage
  il_ak_hest            NUMERIC(12,2),                                          -- Herstellkosten zum Zeitpunkt der Inventuranlage
  il_SerNrString        TEXT                                                    -- Zum Erstellzeitpunkt bekannte Seriennummern (mit Zeilenumbrüchen dazwischen)
 );
--

CREATE OR REPLACE FUNCTION invlag__b_iu() RETURNS TRIGGER AS $$
BEGIN
 new.il_ort:=COALESCE(new.il_ort, '');
 new.il_chnr:=COALESCE(new.il_chnr, '');

 IF (tg_op='INSERT') THEN
   new.il_ort:=trim(both ' ' from new.il_ort);
   new.il_chnr:=trim(both ' ' from new.il_chnr);
 END IF;

 RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER invlag__b_iu
BEFORE INSERT OR UPDATE
ON invlag
FOR EACH ROW
EXECUTE PROCEDURE invlag__b_iu();

CREATE UNIQUE INDEX uniquelaginv ON invlag (il_iv_nr, il_aknr, il_ort, il_chnr, il_dim1, il_dim2, il_dim3);
CREATE INDEX invlag_lg_ort ON invlag(il_ort);

--Artikelbezogener Lagerabgang / Lagermengen berechnen / Abgänge vorschlagen für bestimmten Lagerplatz
CREATE OR REPLACE FUNCTION calc_lagernd_abgang(IN id integer, IN aknr character varying, IN abgangsoll numeric,
                OUT abgangist numeric, OUT sumabgang numeric, OUT perf boolean, OUT abgort character varying, OUT abgchnr character varying) RETURNS record AS $$
 DECLARE rec RECORD;
        sumbisher NUMERIC;
 BEGIN
  perf:=False;
  sumbisher:=0;
  --Lagerort und Bestand fuer Artikel wo groesste Anzahl vorhanden ist
  SELECT lgort, lgchnr, lganztot INTO rec FROM tartikel.art__lag__lg_anztot__by__inputparams(aknr);
  abgort:=rec.lgort;
  abgchnr:=rec.lgchnr;
  --Keine negativen Lagerbestaende
  IF (TSystem.Settings__GetBool('no_neg_lag')) THEN
        --Soviel ist bisher weg von lganztot
        sumbisher:=COALESCE(TSystem.Settings__GetNumeric('_tmp'||id),0);

        --Der Abgang wuerde noch gehen
        IF ((sumbisher+abgangsoll) <= rec.lganztot) THEN
                abgangist:=abgangsoll;
                sumabgang:=ADDSUM(id,abgangist);
                perf:=True;
                --RAISE NOTICE 'Soll:%-Ist:%-GesamtBisher:%-Gesamtdanach:%-Lagernd:%', abgangsoll,abgangist,sumbisher,sumabgang,rec.lganztot;
        ELSE
                --Eine Restmenge ist vorhanden, die koennten wir noch abgehen lassen
                IF (sumbisher < rec.lganztot) THEN
                        abgangist:=rec.lganztot-sumbisher;
                        sumabgang:=ADDSUM(id,abgangist);
                        sumbisher:=COALESCE(TSystem.Settings__GetNumeric('_tmp'||id),0);
                        perf:=FALSE;
                ELSE --Hier ist nichts mehr uebrig auf dem Lagerort
                        abgangIst:=0;
                        sumabgang:=sumbisher;
                        perf:=FALSE;
                END IF;
        END IF;
  ELSE  -- Negative Lagerbestaende erlaubt, uns ist alles egal.
        abgangist:=abgangsoll;
        sumabgang:=ADDSUM(id,abgangsoll);
        perf:=TRUE;
  END IF;
  RETURN;
 END $$ LANGUAGE 'plpgsql' VOLATILE;


-------
CREATE OR REPLACE FUNCTION tlager.wendatchange_checkbestwirksam(_wendat wendat) RETURNS boolean AS $$
   DECLARE _maxloid integer;
           _maxloidwenr integer;
 BEGIN
 --letzte Bewegung des Artikels auf dem Lagerort
        _maxloid :=      max(lo_id)
                   FROM lagerlog
        --WHERE lo_wendat<>wenr -- nicht nehmen, Alle erfassen, da =+! derselben WE und Inventuren miterfasst werden müssen
                  WHERE lo_aknr_new = _wendat.w_aknr AND lo_lgort_new = _wendat.w_lgort AND lo_lgchnr_new = _wendat.w_lgchnr
                    AND lo_dim1_new = _wendat.w_dim1 AND lo_dim2_new  = _wendat.w_dim2  AND lo_dim3_new   = _wendat.w_dim3
                    AND lo_aktion  <> '+'
                    AND lo_aktion  <> '=+'
        AND (lo_aktion <> '-' OR TSystem.Settings__GetBool('no_neg_lag')) -- WA Nur ausschließen wenn Negative Lagerbestände erlaubt
        AND (lo_aktion <> '=-' OR TSystem.Settings__GetBool('no_neg_lag'));

 --letzte gültige Bewegung zum Wareneingang; die letzte Bewegung kann ja bereits durch die vorletzte nicht bestandswirksam geworden sein
        _maxloidwenr
                :=      coalesce(max(lo_id), _maxloid, -1)
                   FROM lagerlog -- _maxloid, -1 für _maxloidwenr IS NULL oder für _maxloid IS NULL AND _maxloidwenr IS NULL
                  WHERE lo_wendat = _wendat.w_wen
        AND lo_aktion NOT LIKE '%!%';
        --AND lo_aktion NOT LIKE '%=%'; -- nicht nehmen, =+ ist erlaubt
 --
        RETURN coalesce(_maxloidwenr >= coalesce(_maxloid, _maxloidwenr), false);
 END $$ LANGUAGE plpgsql;


-------
CREATE OR REPLACE FUNCTION tlager.lifschchange_checkbestwirksam(_lifsch lifsch) RETURNS boolean AS $$
   DECLARE _maxloid integer;
           _maxloidlifsch integer;
 BEGIN
        _maxloid :=      max(lo_id)
                   FROM lagerlog
        -- WHERE lo_lifsch<>lnr -- nicht nehmen, Alle erfassen, da =+! derselben WE und Inventuren miterfasst werden müssen
                  WHERE lo_aknr_new = _lifsch.l_aknr AND lo_lgort_new = _lifsch.l_lgort AND lo_lgchnr_new = _lifsch.l_lgchnr
                    AND lo_dim1_new = _lifsch.l_dim1 AND lo_dim2_new  = _lifsch.l_dim2  AND lo_dim3_new   = _lifsch.l_dim3
                    AND lo_aktion  <> '+'
                    AND lo_aktion  <> '=+'
        AND (lo_aktion <> '-' OR TSystem.Settings__GetBool('no_neg_lag')) -- WA Nur ausschließen wenn Negative Lagerbestände erlaubt
        AND (lo_aktion <> '=-' OR TSystem.Settings__GetBool('no_neg_lag'));

        _maxloidlifsch
                :=      coalesce(max(lo_id), _maxloid, -1)
                   FROM lagerlog -- _maxloid, -1 für maxloidwenr IS NULL oder für _maxloid IS NULL AND maxloidwenr IS NULL
                  WHERE lo_lifsch = _lifsch.l_nr
        AND lo_aktion NOT LIKE '%!%'
        AND (lo_aktion <> '=-' OR NOT TSystem.Settings__GetBool('no_neg_lag')); -- '=-' ist gültiger WA, wenn Negative Lagerbestände erlaubt
 --
        RETURN coalesce(_maxloidlifsch >= coalesce(_maxloid, _maxloidlifsch), false);
 END $$ LANGUAGE plpgsql;
--

CREATE OR REPLACE FUNCTION tlager.chargennummer__vorgabe__halbfertigteile__get(
      _chargennummer varchar,
      _a2_id integer,
      _w_wen integer
  ) RETURNS varchar AS $$
  DECLARE
      _ab_ix varchar;
      _a2_n  integer;
      _ch_n  varchar;
  BEGIN

     -- ersetzt die Chargennummer durch die ID des letzten abgeschlossenen Arbeitsgangs
     -- Format: UE [ABK=... AG=...]

      _a2_n  :=     a2_n FROM ab2 WHERE a2_id = _a2_id;
      _ab_ix := a2_ab_ix FROM ab2 WHERE a2_id = _a2_id;

      IF _a2_n IS null THEN
          _ch_n := _chargennummer;
      ELSE
          _ch_n :=
                      prodat_languages.lang_text( 2569 ) -- UE
                   || ' ['
                   || prodat_languages.lang_text( 105 )  -- ABK
                   || '='
                   || _ab_ix
                   || ' '
                   || prodat_languages.lang_text( 160 )  -- AG
                   || '='
                   || _a2_n
                   || ' '
                   -- muss eindeutig sein, da sonst umbuchen und zusammenlegen die datenstruktur brechen würde - da lg_w_wen verloren gehen würde
                   || prodat_languages.lang_text( 2163 )  -- WE-Nr
                   || '='
                   || _w_wen
                   || ']';
    END IF;

    RETURN upper( _ch_n );

END $$ LANGUAGE plpgsql STABLE;

-- Vorgabe-Charge anhand Parameter, Settings und AC-Settings generieren, #7741
CREATE OR REPLACE FUNCTION TLager.GetChargennummerVorgabe(
    IN _w_wen integer,
    IN _ak_nr varchar,
    IN _ld_id integer,
    IN _w_l_krz varchar,
    IN _w_zug_dat timestamp DEFAULT currenttime(),
    IN _AddDelimiter boolean DEFAULT true,
    IN _GenerateNew boolean DEFAULT true,
    IN _a2_id integer DEFAULT null
    )
    RETURNS varchar(100)
    AS $$
    -- Zustand A: Lagerzugang frei: freie Eingabe von Artikelnummer und Lieferant
    --         B: Lagerzugang auf Bestellung: In Abhängigkeit ld_code wird bei "I" Optional Verwendung von ABK-Index, ansonsten analog Lagerzugang frei über Einstellung Datum/WeNr
    -- _w_wen > Eingang zukünftige WE-Nummer (Datensatz noch nicht vorhanden) > Notwendig, um WE_NR=Charge zu ermöglichen
    DECLARE chnr varchar:= '';
            ac_charge_abk1 boolean;
            ac_charge_wen1 boolean;
            ac_charge_dat1 boolean;
            _w_wen_chnr varchar;
            wennr_anf boolean;
            r_ldsdok record;
            spacer varchar;
  BEGIN
    SELECT ld_code, ld_abk, ld_a2_id INTO r_ldsdok FROM ldsdok WHERE ld_id = _ld_id;

    IF nullif( _a2_id, 0 ) IS null THEN
      -- #16362 Halbfertigeinlagerungen (_a2_id gefüllt) sind immer chargennummernpflichtig! Daher darf nicht mit LEER rausgesprungen werden!!

        IF TSystem.Settings__GetBool('ChkChrgNrVergabe_nur_ChargPflicht') AND NOT (SELECT ak_chnrreq FROM art WHERE ak_nr = _ak_nr) THEN
           --- #10859 [Chargenpflicht] Setting: Chargennummernvergabe nur bei chargenpflichtigen Artikeln (LZ)
           RETURN '';
        END IF;

        IF r_ldsdok.ld_a2_id IS NOT NULL THEN --Auswärtsbearbeitung hat aktuell keine Charge
           RETURN '';
        END IF;

        IF IsDevelopSystem() OR (TSystem.Settings__Get('KUNDE') = 'KREYENBERG') THEN   -- ggf. später für alle freischalten, Praxistest bei KB
                IF (r_ldsdok.ld_code = 'E') THEN -- Freier WE oder WE zu Externen Bestellungen und
                    -- keine Chargenpflicht lt. Bestellvorschlag, #7973
                IF (SELECT BOOL_AND((teinkauf.bestvorschlag__check_chargenpflicht(main_ag_id)).bvs_result IS NULL)
                      FROM (
                            SELECT la_ag_id AS main_ag_id
                              FROM ldsauftg
                              LEFT JOIN auftg AS matpos ON ag_id = la_ag_id
                             WHERE la_ld_id = _ld_id
                            ) AS sub
                        )
                THEN
                    RETURN lang_text(30111); -- 'OHNE' = ohne Charge bestellt
                END IF;
            END IF;
        END IF;
    END IF;

    IF -- Einstellung der Chargenvorgabe kommt aus AC!
       (SELECT true
          FROM artcod JOIN art ON ak_ac = ac_n
         WHERE ak_nr = _ak_nr
           AND (ac_charge_abk OR ac_charge_wen OR ac_charge_dat)
        )
        THEN -- nur wenn ein AC-Setting wahr ist
        SELECT coalesce(ac_charge_abk, false),
               coalesce(ac_charge_wen, false),
               coalesce(ac_charge_dat, false)
          INTO ac_charge_abk1,
               ac_charge_wen1,
               ac_charge_dat1
          FROM artcod JOIN art ON ak_ac = ac_n
         WHERE ak_nr = _ak_nr;
    ELSE -- sonst globale Settings
        SELECT coalesce(TSystem.Settings__GetBool('AbkIsCharge'), false),
               coalesce(TSystem.Settings__GetBool('ChkWenIsCharge'), false),
               coalesce(TSystem.Settings__GetBool('ChkDatIsCharge'), false)
          INTO ac_charge_abk1,
               ac_charge_wen1,
               ac_charge_dat1;
    END IF;

    _w_wen_chnr:='';

    IF ac_charge_wen1 THEN
        _w_wen_chnr   := _w_wen;
    END IF;

    IF _w_wen=-99 THEN --Platzhalter für neue WENR (Lagerzugang tabellarisch)
        _w_wen_chnr:='<WENR>';
    END IF;

    wennr_anf := TSystem.Settings__GetBool('ChkChargeWenNrAnfang');
    IF wennr_anf AND ac_charge_wen1 THEN
        chnr:= 'WENR=' || _w_wen_chnr;
    END IF;

    spacer:= ' '; -- Trenner zwischen Elementen (die nicht NULL sind) durch concat_ws(spacer, ...). Ggf. anpassen.

    IF  -- interne Bestellung mit ABK
        ac_charge_abk1 AND r_ldsdok.ld_code = 'I' AND r_ldsdok.ld_abk > 0
    THEN
        chnr :=     concat_ws(spacer,
                              nullif(chnr, ''),
                              'ABK=' || r_ldsdok.ld_abk
                              );
    ELSIF -- externe Bestellung mit WENR und Adresse
        ac_charge_wen1 AND r_ldsdok.ld_code = 'E' AND NOT ac_charge_dat1
    THEN
        IF wennr_anf THEN
            chnr := concat_ws(spacer,
                              nullif(chnr, ''),
                              'L=' || nullif(_w_l_krz, '')
                              );
        ELSE
            chnr:= concat_ws(spacer,
                IfThen(wennr_anf, 'WENR=', '') || _w_wen_chnr, -- ? ('' || _w_wen), ggf. NULL statt '', wenn komplett ohne w_wen sein soll. Setting unbekannt, evtl. kundenspez. verstecktes Setting
                'L=' || nullif(_w_l_krz, '')
            );
        END IF;
    ELSIF -- ohne Bestellung mit Datum und Adresse
        ac_charge_dat1
    THEN
        chnr:= concat_ws(spacer,
            nullif(chnr, ''),
            'DAT=' || to_char(_w_zug_dat, 'YYMMDD'),
            'L=' || nullif(_w_l_krz, '')
        );
    END IF;

    chnr := TRIM(chnr);
    --- Kombination nicht mit WE-Nr beginnen und kein abk-charge, datum-charge, .... soll noch mal geprüft und getestet werden
    IF (chnr = '') AND ac_charge_wen1 THEN
        chnr:= IFTHEN(wennr_anf, 'WENR=', '') || _w_wen_chnr;
    END IF;

    -- ggf. Ersetzung durch UE-Chargennummer
    chnr := tlager.chargennummer__vorgabe__halbfertigteile__get( chnr, _a2_id, _w_wen );

    IF _AddDelimiter AND chnr <> '' THEN
        chnr := chnr || ' / ';
    END IF;

    RETURN chnr;
  END $$ LANGUAGE plpgsql STABLE;
--

--- #11808 -  Eingabeassistent/Hilfe für Chargen: RunTimeForm (Nr. 29371) setzt formatierten String zusammen
CREATE OR REPLACE FUNCTION tlager.lag__lg_chnr__Customer(IN _wenr VARCHAR, IN _kz VARCHAR, IN _charge VARCHAR, IN _abk VARCHAR) RETURNS VARCHAR AS $$
 DECLARE result    VARCHAR :='';
 BEGIN
    IF CHAR_LENGTH(TRIM(_wenr)) = 0 THEN
       _wenr := null;
    END IF;
    IF CHAR_LENGTH(TRIM(_kz)) = 0 THEN
       _kz := null;
    END IF;
    IF CHAR_LENGTH(TRIM(_charge)) = 0 THEN
       _charge := null;
    END IF;
    /*IF CHAR_LENGTH(TRIM(_abk)) = 0 THEN
       _abk := null;
    END IF;*/
    _abk := null;                    -- keine Entscheidung ist, ob ABK im Chargennummer erscheinen soll

    IF TSystem.Settings__Get('KUNDE') IN ('MATTIS', 'DEMO') THEN
       RETURN concat_ws(' / ', TRIM(_wenr), 'KZ=' || TRIM(_kz), 'CH:' || TRIM(_charge), TRIM(_abk));
    ELSE
       RETURN null;
    END IF;
END $$ LANGUAGE plpgsql STABLE;
--

-- Seriennummern aus Text extrahieren und zeilenweise ausgeben. #10781
CREATE OR REPLACE FUNCTION TLager.lagsernr__text_extract_to_sernr(IN text_to_extract TEXT) RETURNS SETOF VARCHAR AS $$
  DECLARE part VARCHAR;
  BEGIN
    text_to_extract:= regexp_replace(text_to_extract, E'\\s*(,|;|\\t|\\r|\\n)+\\s*', ';', 'g'); -- 1. Ersetze mit Semikolon (steuert dann Trennung) alle von 0-n Leerzeichen umschlossenen 1-k aufeinanderfolgende Zeichen (Komma oder Semikolon oder Tab oder CR oder LF).

    FOR part IN SELECT regexp_split_to_table(text_to_extract, E'\\s*;\\s*') LOOP                -- 2. Trenne String anhand von Semikolon, welches von 0-n Leerzeichen umschlossen sein kann.
        part:= trim(part, '"'' ');                                                              -- 3. Entferne beim Ergebnis äußere " ' und Leerzeichen.
        IF part <> '' THEN
            RETURN NEXT part;
        END IF;
    END LOOP;

    -- Alternative per LANGUAGE sql. Evtl. performanter.
    -- SELECT * FROM (SELECT trim(regexp_split_to_table(regexp_replace(text_to_extract, E'\\s*(,|;|\\t|\\r|\\n)+\\s*', ';', 'g'), E'\\s*;\\s*'), '"'' ') AS part) AS sub WHERE part <> '';
  END $$ LANGUAGE plpgsql IMMUTABLE STRICT;


CREATE OR REPLACE FUNCTION tlager.lagsernr__sernr__frei(
    _sernr varchar,
    _ld_id integer,
    _w_wen integer DEFAULT null
  ) RETURNS boolean AS $$
  DECLARE
    _r record;
    _ld_id2 integer;
    _ld_aknr varchar;
    _ld_kn varchar;
    _ld_aknr2 varchar;
    _ld_kn2 varchar;
  BEGIN

    -- prüft, ob eine Seriennummer zu einer Bestellung und ggf. einem Wareneingang noch frei ist,
    --   d.h. nicht bereits dem gleichen Artikel und Lieferanten zugeordnet
    -- #19765 Ausgebuchte Seriennummern finden bei Duplikatsprüfung keine Berücksichtigung.
    FOR _r IN ( SELECT lgs_id, lgs_w_wen FROM lagsernr WHERE lgs_sernr = _sernr AND lgs_l_nr IS null ) LOOP

      -- wird nur verglichen, wenn _w_wen tatsächlich übergeben wurde
      IF _r.lgs_w_wen = _w_wen THEN
        RETURN false;
      END IF;

      _ld_id2 := coalesce(
        ( SELECT ms_pkey::integer FROM mapsernr WHERE ms_lgs_id = _r.lgs_id AND ms_table = 'ldsdok'::REGCLASS ),
        ( SELECT w_lds_id FROM wendat WHERE w_wen = _r.lgs_w_wen ));

      IF _ld_id2 = _ld_id THEN
        RETURN false;
      END IF;

      SELECT ld_aknr, ld_kn INTO  _ld_aknr,  _ld_kn FROM ldsdok WHERE ld_id =  _ld_id;
      SELECT ld_aknr, ld_kn INTO _ld_aknr2, _ld_kn2 FROM ldsdok WHERE ld_id = _ld_id2;
      IF _ld_aknr = _ld_aknr2 AND _ld_kn = _ld_kn2 THEN
        RETURN false;
      END IF;

    END LOOP;

    RETURN true;

  END $$ LANGUAGE plpgsql STABLE;


  -- Erzeugt eine Vorgabeseriennummer anhand der Sereinnummerbezeichnung und der ID der Bestellposition-
  -- Gibt die lgs_id der erzeugten Sereinnummer zurück, falls diese Operation gelang.
  -- Kommt dagegen null zurück, kann es dafür drei Gründe geben:
  --   1. Die übergebene ld_id gehört zu keiner Bestellposition.
  --   2. Die Seriennummerbezeichnung ist null.
  --   3. Die neue Seriennummer wurde als Duplikat verworfen.
  CREATE OR REPLACE FUNCTION tlager.lagsernr__vorgabesernr__create( _sernr varchar, _ld_id integer ) RETURNS integer AS $$
  DECLARE
      _lgs_id integer;
  BEGIN

      -- _ld_id zeigt ins Leere? Dann gibt es nichts zu tun.
      IF NOT EXISTS( SELECT 1 FROM ldsdok WHERE ld_id = _ld_id ) THEN
        RETURN null;
      END IF;

      -- Speicherung der Vorgabeseriennummer
      INSERT INTO lagsernr( lgs_sernr )
      VALUES              (  _sernr )
      RETURNING lgs_id INTO _lgs_id;

      -- Ablage des Bezugs zur Bestellung und somit Kennzeichnung als Vorgabeseriennummer
      INSERT INTO mapsernr( ms_lgs_id, ms_pkey , ms_table           )
      VALUES              ( _lgs_id  , _ld_id  , 'ldsdok'::REGCLASS );

      -- Die soeben eingefügten Datensätze kann (wegen doppelter Seriennummer) per After-Trigger wieder gelöscht worden sein.
      -- Dies geschieht, wenn diese Seriennummer bereits vorhanden und mit der Bestellung verknüpft ist.
      IF EXISTS( SELECT 1 FROM lagsernr WHERE lgs_id = _lgs_id ) THEN
        RETURN _lgs_id;
      ELSE
        RETURN null;
      END IF;

  END $$ LANGUAGE plpgsql STRICT VOLATILE;
  --

  -- https://redmine.prodat-sql.de/issues/17267
  -- parst Vorgabeseriennummern im CSV-Format
  --   und legt sie in der Tabelle lagsernr ab
  -- speichert Bestellbezug in Tabelle mapsernr
  CREATE OR REPLACE FUNCTION tlager.lagsernr__mapsernr__vorgabesernr__extract_from_text(
      _vorgabesernr text,            -- Vorgabeseriennummern als CSV
      _ld_id integer,                -- Bestellung
      _lg_id integer DEFAULT null,   -- Lagerort
      _w_wen integer DEFAULT null    -- Lagerzugang
    ) RETURNS VOID AS $$
    DECLARE
      _r record;
      _lgs_id integer;
    BEGIN

      FOR _r IN (
          SELECT sub.sernr AS sernr
          FROM ( SELECT DISTINCT tlager.lagsernr__text_extract_to_sernr( _vorgabesernr ) AS sernr) AS sub
          WHERE tlager.lagsernr__sernr__frei( sub.sernr, _ld_id, _w_wen )
      )
      LOOP

        -- Speicherung der Vorgabeseriennummer, ggf. mit Wareneingang und Lagerort
        _lgs_id := tlager.lagsernr__vorgabesernr__create( _r.sernr, _ld_id );

        IF _lgs_id IS NOT null THEN
          IF _lg_id IS NOT null OR _w_wen IS NOT null THEN
            UPDATE lagsernr SET lgs_lg_id = _lg_id, lgs_w_wen = _w_wen WHERE lgs_id = _lgs_id;
          END IF;
        END IF;

      END LOOP;

    END $$ LANGUAGE plpgsql;
--

--
-- #17835 Kundenspezifische Seriennummer-Formatierungsfunktion, nach dem Scan-Ereignis.
-- Diese Funktion wird aus PRODAT aufgerufen, kann aber vom Kunden überschrieben werden. Es kann eine andere Routine hier
-- in diese Funktion beim Kunden eingebaut werden, welche die gescannten Seriennummern verändert, z.b: Matrixcode in SN!
CREATE OR REPLACE FUNCTION z_50_customer.lager__seriennummer_after_scan(_sernr varchar) RETURNS varchar AS $$
  SELECT _sernr; -- gibt die eingehende SN im Standard zurück
$$ LANGUAGE sql STABLE;
--

-- #17835, #18576 Beziehnung zwischen Seriennummern (PA) und Stempeldatensatz herstellen
CREATE OR REPLACE FUNCTION tlager.mapsernr__bdea__add(
    _ab_ix integer,
    _a2_n integer,
    _Seriennummern text,
    _IsAusschuss boolean,
    _ba_asg_id integer = null
  )
  RETURNS TABLE (
    msnb_id integer,
    msnb_sernr varchar,
    msnb_value varchar,
    msnb_status boolean
  ) AS $$
DECLARE
  _ba_id integer;                            -- Stempeldatensatz-ID
  _sn_ag_exists boolean;                     -- Seriennummer bereits mit diesem AG verknüpft
  _lgs_id integer;                           -- Lagerseriennummer-ID
  _ld_id integer;                            -- Produktionsauftrag-ID
  _n timestamp;                              -- Now, quasi jetzt
  _r record;                                 -- jede Seriennummer einzeln
  _cnt integer;                              -- Anzahl der gefertigten Teile (Count)
  _fertigungszeit_je_stueck_h numeric(12,4); -- Fertigungszeit eines Stücks in Stunden
  _auftragszeit_h numeric(12,4);             -- geplante Auftragszeit des AG in Stunden
BEGIN
  -- Zeitstempel zwischenspeichern
  _n := Now();

  -- Stempeldatensatz erzeugen
  INSERT INTO bdea
   (
    ba_minr,
    ba_anf,
    ba_end,
    ba_ix,  --ABK
    ba_op,  --AG
    ba_ks,
    ba_in_rckmeld,
    ba_ende,
    ba_ruest,
    ba_stk,
    ba_auss
   )
  VALUES
   (
    current_user_minr(),
    _n,
    null,
    _ab_ix,
    _a2_n,
    ( SELECT a2_ks FROM ab2 WHERE a2_ab_ix = _ab_ix AND a2_n = _a2_n ),
    False,
    False,
    False,
    0, -- <count SN + Flag ausschuss = false> - wird später per Update gesetzt
    0  -- <count SN + Flag ausschuss = true> - wird später per Update gesetzt
   )
  RETURNING ba_id INTO _ba_id;

  -- Bestellung (PA) finden
  _ld_id := ab_ld_id FROM abk WHERE ab_ix = _ab_ix;

  -- Hilfstabelle erzeugen
  DROP TABLE IF EXISTS temptable__tlager__mapsernr__bdea;
  CREATE TEMPORARY TABLE temptable__tlager__mapsernr__bdea(
    msnb_id integer,
    msnb_sernr VARCHAR,
    msnb_value VARCHAR,
    msnb_status BOOL
  );

  -- alle Seriennummern einzeln verarbeiten
  FOR _r IN (
              SELECT
                z_50_customer.lager__seriennummer_after_scan(sub.sernr) AS sernr
              FROM
                ( SELECT DISTINCT TLager.lagsernr__text_extract_to_sernr( _Seriennummern ) AS sernr ) AS sub
            )
  LOOP
    -- Lagerseriennummer-ID finden
    _lgs_id := lgs_id
      FROM lagsernr
      JOIN mapsernr ON ms_lgs_id = lgs_id
        AND ms_pkey = _ld_id -- Vorgabeseriennummer (= Bestellung/Lagsernr-Bezug)
        AND ms_table = 'ldsdok'::REGCLASS
      WHERE lagsernr.lgs_sernr ILIKE _r.sernr --SN gleich
      LIMIT 1;

    IF _lgs_id IS NULL THEN
      -- Fehler für Funktionsdebugging ausgeben! Keine Exception auslösen, da gesammelte Daten verarbeitet werden!
      RAISE NOTICE 'serial number "%" not found!', _r.sernr;
      -- Fehler als Funktionsergebnis protokollieren
      INSERT INTO temptable__tlager__mapsernr__bdea
       (
        msnb_id,
        msnb_sernr,
        msnb_value,
        msnb_status
       )
      VALUES
       (
        NULL,
        _r.sernr,
        LangText(30214), -- "Produktseriennummer nicht im Produktionsauftrag gefunden."
        false
       );
    ELSE
      -- Seriennummer vorhanden, also buchen, falls nicht schon vorhanden
      _sn_ag_exists := COUNT(*) > 0
        FROM bdea
        JOIN mapsernr ON ms_pkey = ba_id AND ms_table = 'bdea'::REGCLASS
        JOIN lagsernr ON ms_lgs_id = lgs_id
        WHERE lagsernr.lgs_sernr ILIKE _r.sernr
          AND bdea.ba_op = _a2_n;

      IF _sn_ag_exists THEN
        -- Problem für Funktionsdebugging ausgeben! Keine Exception auslösen, da gesammelte Daten verarbeitet werden!
        RAISE NOTICE 'serial number "%" already exist for op "%"!', _r.sernr, _a2_n;
        -- Problem als Funktionsergebnis protokollieren
        INSERT INTO temptable__tlager__mapsernr__bdea
         (
          msnb_id,
          msnb_sernr,
          msnb_value,
          msnb_status
         )
        VALUES
         (
          _lgs_id,
          _r.sernr,
          LangText(30215), --"Seriennummer wurde bereits an dem Arbeitsgang erfasst."
          false
         );
      ELSE
        -- Ablage des Bezuges zum Stempeldatensatz
        INSERT INTO mapsernr
         (
          ms_lgs_id,
          ms_pkey,
          ms_table
         )
        VALUES
         (
          _lgs_id,
          _ba_id,
          'bdea'::REGCLASS
         );

        -- Erfolg als Funktionsergebnis protokollieren
        INSERT INTO temptable__tlager__mapsernr__bdea
         (
          msnb_id,
          msnb_sernr,
          msnb_value,
          msnb_status
         )
        VALUES
         (
          _lgs_id,
          _r.sernr,
          to_char(_n, 'HH24:MI:SS'),
          true
         );
      END IF;
    END IF;
  END LOOP;

  -- Anzahl der Buchungen; je Buchung ein Teil (ein Stück)
  _cnt := count(*)
           FROM temptable__tlager__mapsernr__bdea
          WHERE temptable__tlager__mapsernr__bdea.msnb_status;

  -- Wenn überhaupt Teile gebucht wurden
  IF _cnt > 0 THEN

    -- Menge für den obrigen bdea-Datensatz setzen: ba_stk oder ba_auss
    IF _IsAusschuss THEN
      -- Ausschuss wird im Feld ba_auss vermerkt
      UPDATE bdea
         SET ba_auss = _cnt,
             ba_asg_id = _ba_asg_id
       WHERE ba_id = _ba_id
         AND ba_stk + ba_auss = 0;
    END IF;

    -- Auftragszeit = geplante Zeit * gefertigte Menge / Fertigungsmenge ABK
    SELECT a2_ta * _cnt / coalesce( nullif( ab_st_uf1, 0 ), 1 ) INTO _auftragszeit_h
    FROM ab2
    JOIN abk ON a2_ab_ix = ab_ix
    WHERE a2_ab_ix = _ab_ix AND a2_n = _a2_n;

    -- Fertigungszeit je Stück = geplante Zeit / Fertigungsmenge ABK
    _fertigungszeit_je_stueck_h := _auftragszeit_h / _cnt;

    -- Zeitverbuchung für Fertig- und für Ausschussmeldung
    UPDATE bdea
      SET ba_stk = _cnt,
          ba_stk_time = _fertigungszeit_je_stueck_h,
          ba_efftime = _auftragszeit_h
    WHERE ba_id = _ba_id;

  ELSE
    --Keine Teile gebucht? Dann kann der bdea-Datensatz weg
    DELETE FROM bdea
     WHERE ba_id = _ba_id
       AND ba_stk + ba_auss = 0;
  END IF;

  -- Werte aus dem Temptable als Ergebnis zurückgeben
  RETURN QUERY
         SELECT * FROM temptable__tlager__mapsernr__bdea ORDER BY temptable__tlager__mapsernr__bdea.msnb_sernr;

  DROP TABLE temptable__tlager__mapsernr__bdea;

END $$ LANGUAGE plpgsql;
--

-- #21116 SN-AG-Tracking: Mapping von SN mit Arbeitsgang für SN-spezifische Arbeitsgänge
CREATE OR REPLACE FUNCTION tlager.mapsernr__ab2__add(
        IN  _a2_id                     integer,
        IN  _lgs_id                    integer,
        IN  _banned_a2_n_schrittweite  integer DEFAULT null, -- Arbeitgangnummern in dieser Schrittweite dürfen nicht einzelnen SN zugeordnet werden -> null: alle Arbeitsgangnummern sind zulässig
        OUT _ms_id                     integer               -- SN-Mapping-ID bei erfolgreichem Mapping
      )
    RETURNS integer
    AS $$
    BEGIN

      -- Prüfung ob AG existiert
      IF NOT EXISTS(SELECT 1 from ab2 WHERE a2_id = _a2_id) THEN
          RAISE EXCEPTION 'xtt4027'; -- Arbeitsgang existiert nicht.
      END IF;

      -- Prüfung ob SN existiert und als Vorgabe-SN zu ABK des AGs definiert ist
      IF NOT EXISTS(
          SELECT 1
            FROM ab2
            JOIN ldsdok ON ab2.a2_ab_ix = ldsdok.ld_abk
            JOIN mapsernr ON mapsernr.ms_pkey  = ldsdok.ld_id
                         AND mapsernr.ms_table = 'ldsdok'::REGCLASS
           WHERE mapsernr.ms_lgs_id = _lgs_id
             AND ab2.a2_id = _a2_id
          ) THEN
        RAISE EXCEPTION 'xtt35260'; -- Die Seriennummer ist keine Vorgabe-Seriennummer des zum Arbeitsgang gehörenden Produktionsauftrages.
      END IF;

      -- Wenn Setting aktiv und Arbeitsgangnummer enspricht nicht der Standardvorgabe (z.B. 10er, 100er Schritte...),
      -- dann wird AG als Nacharbeit angesehen und ein individuelles Mapping je SN ist zulässig
      IF coalesce( mod( (SELECT a2_n FROM ab2 WHERE a2_id = _a2_id), _banned_a2_n_schrittweite
                      ) = 0,
                   false ) THEN
        RAISE EXCEPTION 'xtt35261'; -- Dem Arbeitsgang darf aufgrund seiner Arbeitsgangnummer keine Seriennummer zugeordnet werden.
      ELSE
        -- Mapping von SN zu AG setzen
        INSERT INTO mapsernr ( ms_lgs_id, ms_table       , ms_pkey )
               VALUES        ( _lgs_id  , 'ab2'::REGCLASS, _a2_id  )
               RETURNING ms_id INTO _ms_id;
      END IF;

    END $$ LANGUAGE plpgsql;
--

-- #18078 Temporäre Tabelle für die Abfragen der Lagerorte für den ABK-bezogenen Lagerabgang
CREATE TABLE tlager.lagerabgang__abk__temp (
  labk_id serial,
  labk_abix integer NOT null,                    -- ABK, auf die ausgelagert werden soll
  labk_agid integer NOT null,                    -- Eintrag der Materialliste
  labk_transaction_timestamp timestamp NOT null, -- Zeitstempel der Transaktion
  labk_nr_abfrage integer,                       -- laufende Abfrage Nummer, um verschiedenen Abfragen innerhalb einer Transaktion unterscheiden zu können
  labk_lgid integer,                             -- ID des betroffenen Lagerorts
  labk_lagab_uf1 numeric NOT null,               -- bislang innerhalb der Abfrage reservierte Menge
  labk_isAlternativArtikel boolean               -- handelt es sich um einen Alternativartikel?
);


-- Erzeugt eine Vorgabeseriennummer anhand der Sereinnummerbezeichnung und der ID der Bestellposition-
  -- Gibt die lgs_id der erzeugten Sereinnummer zurück, falls diese Operation gelang.
  -- Kommt dagegen null zurück, kann es dafür drei Gründe geben:
  --   1. Die übergebene ld_id gehört zu keiner Bestellposition.
  --   2. Die Seriennummerbezeichnung ist null.
  --   3. Die neue Seriennummer wurde als Duplikat verworfen.
  CREATE OR REPLACE FUNCTION tlager.lagsernr__vorgabesernr__create( _sernr varchar, _ld_id integer ) RETURNS integer AS $$
  DECLARE
      _lgs_id integer;
  BEGIN

      -- _ld_id zeigt ins Leere? Dann gibt es nichts zu tun.
      IF NOT EXISTS( SELECT 1 FROM ldsdok WHERE ld_id = _ld_id ) THEN
        RETURN null;
      ENd IF;

      -- Speicherung der Vorgabeseriennummer, ggf. mit Wareneingang und Lagerort
      INSERT INTO lagsernr( lgs_sernr )
      VALUES              (  _sernr )
      RETURNING lgs_id INTO _lgs_id;

      -- Ablage des Bezugs zur Bestellung und somit Kennzeichnung als Vorgabeseriennummer
      INSERT INTO mapsernr( ms_lgs_id, ms_pkey , ms_table)
      VALUES              ( _lgs_id, _ld_id, 'ldsdok'::REGCLASS);

      -- Die soeben eingefügten Datensätze kann (wegen doppelter Seriennummer) per After-Trigger wieder gelöscht worden sein.
      IF EXISTS( SELECT 1 FROM lagsernr WHERE lgs_id = _lgs_id ) THEN
        RETURN _lgs_id;
      ELSE
        RETURN null;
      END IF;

  END $$ LANGUAGE plpgsql STRICT VOLATILE;
--